
/**
 * Die Klasse DHSimpleHashSet verwaltet eine Menge von Objekten
 * mittels einer Hashtabelle mit double hashing.
 *
 * Der Wert null ist als Eintrag nicht zulssig.
 *
 * @author Clemens Bruckmann
 *
 * @version 2005-03-02
 */
public class DHSimpleHashSet {

   /**
    * Die Hashtabelle.
    */
   protected Object[] hashtabelle;

   /**
    * Dient der Markierung gelschter Eintrge,
    * damit im Kollisionspfad weitergesucht wird.
    */
   protected static final Object WIEDERFREI = new Object();

   /**
    * Die Anzahl der enthaltenen Elemente.
    */
   protected int size = 0;

   /**
    * Konstruktor.
    * Legt die anfngliche Gre der Hashtabelle mit 101 fest.
    */
   public DHSimpleHashSet() {
      this(101);
   }

   /**
    * Konstruktor.
    *
    * @param capacity die anfngliche Gre der Hashtabelle.
    */
   public DHSimpleHashSet(int capacity) {
      hashtabelle = new Object[capacity];
   }

   /**
    * Fgt das angegebene Objekt in diese Menge ein.
    * Falls es bereits enthalten war, bleibt die Menge unverndert.
    *
    * @return true  falls das eingefgte Objekt vorher nicht enthalten war;
    *         false falls das Objekt nicht eingefgt wurde,
    *               da es bereits enthalten war.
    *
    * @throws IllegalArgumentException falls das als Parameter
    *                                  angegebene Objekt null ist.
    */
   public boolean add(Object o) {
      if (o == null) throw new IllegalArgumentException();
      if (1.0 * size / hashtabelle.length >= 0.68) {
         /* Die Hashtabelle ist ziemlich voll, wir fhren ein Rehashing durch */
         rehash();
      }
      int platz = findeTabellenplatz(o);
      if (platz == -1) {
         /* sollte nicht vorkommen (wir fhren oben bereits bei einem      */
         /* Fllfaktor von 68% ein Rehashing durch)                        */
         /* Die Hashtabelle ist voll, wir mssen ein Rehashing durchfhren */
         rehash();
         platz = findeTabellenplatz(o);
      }
      if (hashtabelle[platz] == null || hashtabelle[platz] == WIEDERFREI) {
         hashtabelle[platz] = o;
         size = size + 1;
         return true;      /* erfolgreich eingefgt */
      } else {
         return false;     /* war bereits enthalten */
      }
   }

   /**
    * Gibt an, ob das angegebene Objekt in dieser Menge enthalten ist.
    *
    * @return true falls das angegebene Objekt in dieser Menge enthalten ist.
    */

   public boolean contains(Object o) {
      /* null kann nicht eingefgt werden und kann daher nicht enthalten sein */
      if (o == null) return false;
      int platz = findeTabellenplatz(o);
      return platz != -1 &&
             hashtabelle[platz] != null &&
             hashtabelle[platz] != WIEDERFREI;
   }

   /**
    * Entfernt das angegebene Objekt aus dieser Menge, falls es enthalten ist.
    *
    * @return true falls das Objekt vorher enthalten war.
    */
   
   public boolean remove(Object o) {
      /* null kann nicht eingefgt werden und kann daher nicht enthalten sein */
      if (o == null) return false;
      int platz = findeTabellenplatz(o);
      if (platz == -1) return false;     /* nicht enthalten (und Hashtabelle voll) */
      if (hashtabelle[platz] != null && hashtabelle[platz] != WIEDERFREI) {
         hashtabelle[platz] = WIEDERFREI;
         size = size - 1;
         return true;      /* gefunden und entfernt */
      } else {
         return false;     /* nicht enthalten */
      }
   }

   /**
    * Vergrert die Kapazitt der Hashtabelle.
    */
 protected void rehash() {
      /* wir nehmen hier als neue Gre die doppelte; das ist NICHT optimal! */
      /* Besser wre es, eine Primzahl in dieser Grenordnung zu suchen.    */
      DHSimpleHashSet neueTabelle = new DHSimpleHashSet(hashtabelle.length * 2);
      /* wir fgen jedes Element aus der bestehenden Tabelle in die neue ein */
      for (int i = 0; i < hashtabelle.length; ++i) {
         if (hashtabelle[i] != null && hashtabelle[i] != WIEDERFREI) {
            neueTabelle.add(hashtabelle[i]);
         }
      }
      this.hashtabelle = neueTabelle.hashtabelle;
   }

   /**
    * Gibt den Tabellenplatz an, an welchem das angegebene Objekt steht
    * bzw. stehen msste.
    *
    * @return falls das Objekt enthalten ist: seinen Index in der Hashtabelle;
    *         falls das Objekt nicht enthalten ist: den Platz, an den es
    *            eingefgt werden knnte bzw. -1 falls die Tabelle voll ist.
    */
 
   protected int findeTabellenplatz(Object o) {
      /* Die hier verwendete Kollisionsstrategie ist */
      /* das linear probing, das ist NICHT optimal!  */
      int anfangsTabellenplatz = Math.abs(o.hashCode() % hashtabelle.length);
      int ersteMglichePosition = -1;
      boolean anfang = true;
      for (int tabellenplatz = anfangsTabellenplatz;
           tabellenplatz != anfangsTabellenplatz || anfang;
           tabellenplatz = (tabellenplatz + 1) % hashtabelle.length) {
         if (hashtabelle[tabellenplatz] == null) {
            /* Kollisionspfad zu Ende; gesuchtes Objekt nicht enthalten */
            if (ersteMglichePosition == -1) ersteMglichePosition = tabellenplatz;
            return ersteMglichePosition;
         } else if (hashtabelle[tabellenplatz] == WIEDERFREI) {
            /* muss im Kollisionspfad weitersuchen */
            if (ersteMglichePosition == -1) ersteMglichePosition = tabellenplatz;
         } else if (hashtabelle[tabellenplatz].equals(o)) {
            /* gefunden! */
            return tabellenplatz;
         }  /* muss im Kollisionspfad weitersuchen */
         anfang = false;
      }
      /* nicht enthalten (und Hashtabelle voll) */
      return -1;
   }

}
