
/**
 * Das Interface Graph spezifiziert eine Datenstruktur zur Speicherung
 * eines Graphen ohne Mehrfachkanten.
 *
 * Die Anzahl der Knoten ist unvernderlich.
 * Die Knoten werden mit int-Werten von 0 bis ((Anzahl der Knoten) - 1)
 * bezeichnet.
 *
 * Jede Klasse, die dieses Interface implementiert, soll zumindest
 * folgende drei Konstruktoren aufweisen:
 *
 * "Konstruktor 1"
 *    ein Konstruktor mit zwei Parametern:
 *    einem Parameter vom Typ int,
 *    mit welchem die Anzahl der Knoten dieses Graphen festgelegt wird,
 *    und einem Parameter vom Typ boolean, der angibt, ob der Graph
 *    gerichtet ist oder nicht,
 *
 * "Konstruktor 2"
 *    ein Konstruktor mit einem Parameter vom Typ Graph,
 *    welcher das "Kopieren" smtlicher in diesem Interface festgelegten
 *    Eigenschaften des angegebenen Graphen bewirkt (Anzahl der Knoten,
 *    Benennung der Knoten, gerichtet oder nicht, Kanten samt Kantengewichten),
 *
 * "Konstruktor 3"
 *    ein Konstruktor mit einem Parameter vom Typ org.xml.sax.InputSource,
 *    welcher die Eigenschaften des Graphen aus einem XML-Dokument im XGMML-Format
 *    (siehe http://www.cs.rpi.edu/~puninj/XGMML/ ) einliest.
 *
 * @author Clemens Bruckmann
 *
 * @version 2005-05-25
 */
public interface Graph {

   // Methoden betreffend die Knoten (vertices)

   /**
    * Liefert die Anzahl der Knoten. Die Anzahl der Knoten ist unvernderlich.
    */
   public int getNumberOfVertices();

   /**
    * Legt eine Bezeichnung fr diesen Knoten fest.
    * Die alte Bezeichnung des Knotens wird berschrieben.
    *
    * @param vertex der Knoten, dessen Bezeichnung festgelegt werden soll.
    * @param label  die Bezeichnung, die der angegebene Knoten erhalten soll.
    *
    * @throws IndexOutOfBoundsException falls vertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public void setVertexLabel(int vertex, String label);

   /**
    * Liefert die Bezeichnung des angegebenen Knotens.
    *
    * @param vertex der Knoten, dessen Bezeichnung ermittelt werden soll.
    *
    * @throws IndexOutOfBoundsException falls vertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public String getVertexLabel(int vertex);


   // Methoden betreffend die Kanten (edges)

   /**
    * Gibt an, ob der Graph gerichtet ist oder nicht.
    * Diese Eigenschaft ist unvernderlich.
    */
   public boolean directed();

   /**
    * Fgt eine Kante in den Graphen ein bzw. setzt das Kantengewicht einer
    * bestehenden Kante neu fest.
    *
    * @param fromVertex der Knoten, von welchem die Kante ausgeht
    * @param toVertex   der Knoten, zu welchem die Kante hinfhrt
    * @param weight     das Kantengewicht
    *
    * @throws IndexOutOfBoundsException falls fromVertex oder toVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public void setEdge(int fromVertex, int toVertex, double weight);

   /**
    * Meldet, ob der Graph die angegebene Kante enthlt.
    *
    * @param fromVertex der Knoten, von welchem die Kante ausgehen soll
    * @param toVertex   der Knoten, zu welchem die Kante hinfhren soll
    *
    * @throws IndexOutOfBoundsException falls fromVertex oder toVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public boolean containsEdge(int fromVertex, int toVertex);

   /**
    * Liefert das Kantengewicht einer Kante.
    * Existiert die angegebene Kante nicht, so wirft die Methode
    * eine IllegalStateException.
    *
    * @param fromVertex der Knoten, von welchem die Kante ausgeht
    * @param toVertex   der Knoten, zu welchem die Kante hinfhrt
    *
    * @throws IndexOutOfBoundsException falls fromVertex oder toVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    * @throws IllegalStateException     falls die angegebene Kante nicht existiert.
    */
   public double getEdgeWeight(int fromVertex, int toVertex);

   /**
    * Entfernt die angegebene Kante aus dem Graphen.
    *
    * @param fromVertex der Knoten, von welchem die Kante ausgeht
    * @param toVertex   der Knoten, zu welchem die Kante hinfhrt
    *
    * @return true falls die Kante vorher enthalten war.
    *
    * @throws IndexOutOfBoundsException falls fromVertex oder toVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public boolean removeEdge(int fromVertex, int toVertex);

   /**
    * Das Interface Edge dient dazu, Angaben ber eine Kante zu machen.
    *
    * Es findet nur im Zusammenhang mit dem Kanten-Iterator Verwendung.
    *
    * @author Clemens Bruckmann
    *
    * @version 2005-05-25
    */
   public interface Edge {

      /**
       * Gibt den Knoten an, von welchem diese Kante ausgeht.
       */
      public int getFromVertex();

      /**
       * Gibt den Knoten an, zu welchem diese Kante hinfhrt.
       */
      public int getToVertex();

      /**
       * Gibt das Kantengewicht dieser Kante an.
       */
      public double getWeight();

   }

   /**
    * Liefert einen Iterator ber alle Kanten dieses Graphen.
    * Die Reihenfolge, in welcher diese Kanten zu liefern sind,
    * ist nicht allgemein nher spezifiziert.
    */
   public java.util.Iterator<Graph.Edge> edgeIterator();

   /**
    * Liefert einen Iterator ber alle Kanten, die vom angegebenen
    * Knoten ausgehen.
    * Die Reihenfolge, in welcher diese Kanten zu liefern sind,
    * ist nicht allgemein nher spezifiziert.
    *
    * @param fromVertex der Knoten, dessen ausgehende Kanten
    *                   der Iterator liefern soll.
    *
    * @throws IndexOutOfBoundsException falls fromVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public java.util.Iterator<Graph.Edge> edgeIterator(int fromVertex);


   // Weitere Methoden

   /**
    * Gibt diesen Graphen als XGMML-Dokument zurck.
    *
    * Die Methode toXgmml() gibt ein XGMML-Dokument als String zurck;
    * die Methode, die die Methode toXgmml() aufruft, kann dann
    * dieses Dokument z. B. in eine Datei speichern.
    */
   public String toXgmml();

   public void printDfsAnalyse(int fromVertex1, int toVertex1);
}
