
/**
 * Die Klasse AdjMatrixGraph stellt einen Graphen ohne Mehrfachkanten dar.
 * Die Implementierung erfolgt mittels einer Adjazenzmatrix.
 *
 * Die Anzahl der Knoten ist unvernderlich.
 * Die Knoten werden mit int-Werten von 0 bis ((Anzahl der Knoten) - 1)
 * bezeichnet.
 *
 * @author Clemens Bruckmann
 *
 * @version 2005-05-25
 */
public class AdjMatrixGraph extends AbstractGraph implements java.io.Serializable {

   /**
    * Legt fest, ob dieser Graph gerichtet ist oder nicht.
    */
   protected boolean directed;

   /**
    * Die Namen der Knoten.
    */
   protected String[] v;

   /**
    * Symbolisiert eine nicht existente Kante (Kantengewicht unendlich).
    */
   protected static final double NOEDGE = Double.POSITIVE_INFINITY;

   /**
    * Die Kanten(gewichte).
    */
   protected double[][] e;

   // Konstruktoren

   /**
    * Konstruktor. Legt einen Graphen ohne Kanten an.
    *
    * @param numberOfVertices die Anzahl der Knoten dieses Graphen.
    * @param directed         false falls der Graph ungerichtet ist,
    *                         true  falls der Graph gerichtet ist.
    */
   public AdjMatrixGraph(int numberOfVertices, boolean directed) {
      if (numberOfVertices < 0) {
         throw new IllegalArgumentException("Die Anzahl der Knoten darf nicht negativ sein.");
      }
      this.directed = directed;
      v = new String[numberOfVertices];
      e = new double[numberOfVertices][numberOfVertices];
      for (int i = 0; i < numberOfVertices; ++i) {
         v[i] = "v" + i;
         for (int j = 0; j < numberOfVertices; ++j) {
            e[i][j] = NOEDGE;
         }
      }
   }

   /**
    * Konstruktor.
    *
    * @param g der Graph, dessen Knoten und Kanten kopiert werden sollen.
    */
   public AdjMatrixGraph(Graph g) {
      this.directed = g.directed();
      int numberOfVertices = g.getNumberOfVertices();
      v = new String[numberOfVertices];
      e = new double[numberOfVertices][numberOfVertices];
      for (int i = 0; i < numberOfVertices; ++i) {
         v[i] = g.getVertexLabel(i);
         for (int j = 0; j < numberOfVertices; ++j) {
            e[i][j] = NOEDGE;
         }
      }
      for (java.util.Iterator<Graph.Edge> it = g.edgeIterator(); it.hasNext(); ) {
         Graph.Edge edge = it.next();
         e[edge.getFromVertex()][edge.getToVertex()] = edge.getWeight();
         if (!directed) {
            // Ist der Graph ungerichtet, muss auch die Gegenkante eingetragen werden
            e[edge.getToVertex()][edge.getFromVertex()] = edge.getWeight();
         }
      }
   }

   /**
    * Konstruktor.
    *
    * @param is das einzulesende XGMML-Dokument.
    */
   public AdjMatrixGraph(org.xml.sax.InputSource is)
                        throws java.io.IOException, ParseException {
      /* Dieser Konstruktor stellt ziemlich strikte Anforderungen an   */
      /* die XGMML-Datei, z. B. muss jeder Knoten im Attribut id einen */
      /* nichtnegativen int-Wert aufweisen etc.                        */
      /* Fr "ernstzunehmende" Einsatzzwecke wre dieser Konstruktor   */
      /* daher viel zu wenig robust programmiert!                      */
      try {
         javax.xml.parsers.DocumentBuilderFactory dbf = javax.xml.parsers.DocumentBuilderFactory.newInstance();
         dbf.setValidating(true);
         javax.xml.parsers.DocumentBuilder db = dbf.newDocumentBuilder();
         org.w3c.dom.Document doc = db.parse(is);
         org.w3c.dom.Element rootElement = doc.getDocumentElement();
         if (rootElement.getTagName().equals("graph")) {
            directed = (rootElement.getAttribute("directed").equals("1"));
            org.w3c.dom.NodeList nodeNodeList = rootElement.getElementsByTagName("node");
            int maxNodeId = Integer.MIN_VALUE;
            for (int i = 0; i < nodeNodeList.getLength(); ++i) {
               org.w3c.dom.Element nodeElement = (org.w3c.dom.Element) nodeNodeList.item(i);
               int nodeId = Integer.parseInt(nodeElement.getAttribute("id"));
               if (maxNodeId < nodeId) maxNodeId = nodeId;
            }
            int numberOfVert = maxNodeId + 1;
            v = new String[numberOfVert];
            for (int i = 0; i < nodeNodeList.getLength(); ++i) {
               org.w3c.dom.Element nodeElement = (org.w3c.dom.Element) nodeNodeList.item(i);
               int nodeId = Integer.parseInt(nodeElement.getAttribute("id"));
               String nodeLabel = nodeElement.getAttribute("label");
               v[nodeId] = nodeLabel;
            }
            nodeNodeList = null;
            e = new double[numberOfVert][numberOfVert];
            for (int i = 0; i < numberOfVert; ++i) {
               if (v[i] == null || v[i].equals("")) v[i] = "v" + i;
               for (int j = 0; j < numberOfVert; ++j) {
                  e[i][j] = NOEDGE;
               }
            }
            org.w3c.dom.NodeList edgeNodeList = rootElement.getElementsByTagName("edge");
            for (int i = 0; i < edgeNodeList.getLength(); ++i) {
               org.w3c.dom.Element edgeElement = (org.w3c.dom.Element) edgeNodeList.item(i);
               int fromVertex = Integer.parseInt(edgeElement.getAttribute("source"));
               int toVertex   = Integer.parseInt(edgeElement.getAttribute("target"));
               String weightString = edgeElement.getAttribute("weight");
               double weight;
               if (weightString.equals("")) {
                  weight = 1;
               } else {
                  weight = Double.parseDouble(weightString);
               }
               this.setEdge(fromVertex, toVertex, weight);
            }
         } else {
            throw new IllegalArgumentException("Wurzelelement heit " +
                             rootElement.getTagName() + " und nicht \"graph\"");
         }
      } catch(org.xml.sax.SAXParseException e) {
         throw new ParseException("Parser meldet ein Problem in Datei " +
                  e.getSystemId() + ", Zeile " + e.getLineNumber() +
                  ", Spalte " + e.getColumnNumber() + ":\n       " +
                  e.getMessage(), e);
      } catch(org.xml.sax.SAXException e) {
         throw new ParseException(e.getMessage(), e);
      } catch(javax.xml.parsers.ParserConfigurationException e) {
         throw new ParseException(e.getMessage(), e);
      }
   }

   // Methoden betreffend die Knoten (vertices)

   /**
    * Liefert die Anzahl der Knoten. Die Anzahl der Knoten ist unvernderlich.
    */
   public int getNumberOfVertices() {
      return v.length;
   }

   /**
    * Legt eine Bezeichnung fr diesen Knoten fest.
    * Die alte Bezeichnung des Knotens wird berschrieben.
    *
    * @param vertex der Knoten, dessen Bezeichnung festgelegt werden soll.
    * @param label  die Bezeichnung, die der angegebene Knoten erhalten soll.
    *
    * @throws IndexOutOfBoundsException falls vertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public void setVertexLabel(int vertex, String label) {
      v[vertex] = label;
   }

   /**
    * Liefert die Bezeichnung des angegebenen Knotens.
    *
    * @param vertex der Knoten, dessen Bezeichnung ermittelt werden soll.
    *
    * @throws IndexOutOfBoundsException falls vertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public String getVertexLabel(int vertex) {
      return v[vertex];
   }


   // Methoden betreffend die Kanten (edges)

   /**
    * Gibt an, ob der Graph gerichtet ist oder nicht.
    * Diese Eigenschaft ist unvernderlich.
    */
   public boolean directed() {
      return directed;
   }

   /**
    * Fgt eine Kante in den Graphen ein bzw. setzt das Kantengewicht einer
    * bestehenden Kante neu fest.
    *
    * @param fromVertex der Knoten, von welchem die Kante ausgeht
    * @param toVertex   der Knoten, zu welchem die Kante hinfhrt
    * @param weight     das Kantengewicht
    *
    * @throws IndexOutOfBoundsException falls fromVertex oder toVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public void setEdge(int fromVertex, int toVertex, double weight) {
      if (weight == NOEDGE) {
         throw new IllegalArgumentException("Diese Implementierung untersttzt leider " +
                                            "das Kantengewicht " + NOEDGE + " nicht!");
      }
      e[fromVertex][toVertex] = weight;
      if (!directed) {
         // Ist der Graph ungerichtet, muss auch die Gegenkante eingetragen werden
         e[toVertex][fromVertex] = weight;
      }
   }

   /**
    * Meldet, ob der Graph die angegebene Kante enthlt.
    *
    * @param fromVertex der Knoten, von welchem die Kante ausgehen soll
    * @param toVertex   der Knoten, zu welchem die Kante hinfhren soll
    *
    * @throws IndexOutOfBoundsException falls fromVertex oder toVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public boolean containsEdge(int fromVertex, int toVertex) {
     if (e[fromVertex][toVertex] == NOEDGE) {
       return false;
     }
     return true;
   }

   /**
    * Liefert das Kantengewicht einer Kante.
    * Existiert die angegebene Kante nicht, so wirft die Methode
    * eine IllegalStateException.
    *
    * @param fromVertex der Knoten, von welchem die Kante ausgeht
    * @param toVertex   der Knoten, zu welchem die Kante hinfhrt
    *
    * @throws IndexOutOfBoundsException falls fromVertex oder toVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    * @throws IllegalStateException     falls die angegebene Kante nicht existiert.
    */
   public double getEdgeWeight(int fromVertex, int toVertex) {
     if (e[fromVertex][toVertex] == NOEDGE) {
       throw new IllegalStateException("Eine Kante von " + fromVertex + " nach " + toVertex + " existiert nicht!");
     }
     return e[fromVertex][toVertex];
   }

   /**
    * Entfernt die angegebene Kante aus dem Graphen.
    *
    * @param fromVertex der Knoten, von welchem die Kante ausgeht
    * @param toVertex   der Knoten, zu welchem die Kante hinfhrt
    *
    * @return true falls die Kante vorher enthalten war.
    *
    * @throws IndexOutOfBoundsException falls fromVertex oder toVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public boolean removeEdge(int fromVertex, int toVertex) {
     if (e[fromVertex][toVertex] == NOEDGE) {
       return false;
     }
     e[fromVertex][toVertex] = NOEDGE;
     if (!directed) {
       // Ist der Graph ungerichtet, muss auch die Gegenkante eingetragen werden
       e[toVertex][fromVertex] = NOEDGE;
     }
     return true;
   }

   /**
    * Die innere Klasse Edge dient dazu, Angaben ber eine Kante zu machen.
    *
    * Sie findet nur im Zusammenhang mit dem Kanten-Iterator Verwendung.
    *
    * @author Clemens Bruckmann
    *
    * @version 2005-05-25
    */
   protected class Edge implements Graph.Edge {

      /**
       * Der Knoten, von dem diese Kante ausgeht.
       */
      protected int fromVertex;

      /**
       * Der Knoten, zu dem diese Kante hinfhrt.
       */
      protected int toVertex;

      /**
       * Das Kantengewicht dieser Kante.
       */
      protected double weight;

      /**
       * Konstruktor.
       *
       * @param fromVertex Der Knoten, von dem diese Kante ausgeht.
       * @param toVertex   Der Knoten, zu dem diese Kante hinfhrt.
       * @param weight     Das Kantengewicht dieser Kante.
       */
      protected Edge(int fromVertex, int toVertex, double weight) {
         this.fromVertex = fromVertex;
         this.toVertex   = toVertex;
         this.weight     = weight;
      }

      /**
       * Gibt den Knoten an, von welchem diese Kante ausgeht.
       */
      public int getFromVertex() {
         return fromVertex;
      }

      /**
       * Gibt den Knoten an, zu welchem diese Kante hinfhrt.
       */
      public int getToVertex() {
         return toVertex;
      }

      /**
       * Gibt das Kantengewicht dieser Kante an.
       */
      public double getWeight() {
         return weight;
      }

   }

   /**
    * Die innere Klasse OneEdgeIterator stellt einen Iterator ber alle
    * Kanten, die von einem bestimmten Knoten ausgehen, dar.
    *
    * @author Clemens Bruckmann
    *
    * @version 2005-05-25
    */
   protected class OneEdgeIterator implements java.util.Iterator<Graph.Edge> {

      /**
       * Der Knoten, dessen ausgehende Kanten zu liefern sind.
       */
      protected final int fromVertex;

      /**
       * Der Knoten, zu welchem die nchste zu liefernde Kante hinfhrt.
       */
      protected int toVertex;

      /**
       * Ermittelt den Knoten, zu welchem die nchste zu liefernde Kante hinfhrt.
       */
      protected void moveToNextToVertex() {
         // wir berspringen alle nicht existierenden Kanten (Kantenwert NOEDGE)
         for (++toVertex; toVertex < v.length && e[fromVertex][toVertex] == NOEDGE; ++toVertex);
      }

      /**
       * Konstruktor.
       *
       * @throws IndexOutOfBoundsException falls fromVertex auerhalb des zulssigen Bereichs liegt
       *                                   (0 .. getNumberOfVertices()-1).
       */
      protected OneEdgeIterator(int fromVertex) {
         this.fromVertex = fromVertex;
         toVertex = -1;
         moveToNextToVertex();
      }

      public Graph.Edge next() {
         if (toVertex >= v.length) {
            throw new java.util.NoSuchElementException();
         }
         Graph.Edge edge = new Edge(fromVertex, toVertex, e[fromVertex][toVertex]);
         moveToNextToVertex();
         return edge;
      }

      public boolean hasNext() {
         return (toVertex < v.length);
      }

      public void remove() {
         throw new UnsupportedOperationException();
      }
   }

   /**
    * Liefert einen Iterator ber alle Kanten, die vom angegebenen
    * Knoten ausgehen.
    *
    * @param fromVertex der Knoten, dessen ausgehende Kanten
    *                   der Iterator liefern soll.
    *
    * @throws IndexOutOfBoundsException falls fromVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public java.util.Iterator<Graph.Edge> edgeIterator(int fromVertex) {
      return new OneEdgeIterator(fromVertex);
   }

}
