
/**
 * Die abstrakte Klasse AbstractGraph enthlt allgemeine Methoden betreffend
 * Graphen ohne Mehrfachkanten.
 *
 * Die Anzahl der Knoten ist unvernderlich.
 * Die Knoten werden mit int-Werten von 0 bis ((Anzahl der Knoten) - 1)
 * bezeichnet.
 *
 * @author Clemens Bruckmann
 *
 * @version 2005-05-25
 */
public abstract class AbstractGraph implements Graph {

   // Methoden betreffend die Knoten (vertices)

   /**
    * Liefert die Anzahl der Knoten. Die Anzahl der Knoten ist unvernderlich.
    */
   public abstract int getNumberOfVertices();

   /**
    * Legt eine Bezeichnung fr diesen Knoten fest.
    * Die alte Bezeichnung des Knotens wird berschrieben.
    *
    * @param vertex der Knoten, dessen Bezeichnung festgelegt werden soll.
    * @param label  die Bezeichnung, die der angegebene Knoten erhalten soll.
    *
    * @throws IndexOutOfBoundsException falls vertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public abstract void setVertexLabel(int vertex, String label);

   /**
    * Liefert die Bezeichnung des angegebenen Knotens.
    *
    * @param vertex der Knoten, dessen Bezeichnung ermittelt werden soll.
    *
    * @throws IndexOutOfBoundsException falls vertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public abstract String getVertexLabel(int vertex);


   // Methoden betreffend die Kanten (edges)

   /**
    * Gibt an, ob der Graph gerichtet ist oder nicht.
    * Diese Eigenschaft ist unvernderlich.
    */
   public abstract boolean directed();

   /**
    * Fgt eine Kante in den Graphen ein bzw. setzt das Kantengewicht einer
    * bestehenden Kante neu fest.
    *
    * @param fromVertex der Knoten, von welchem die Kante ausgeht
    * @param toVertex   der Knoten, zu welchem die Kante hinfhrt
    * @param weight     das Kantengewicht
    *
    * @throws IndexOutOfBoundsException falls fromVertex oder toVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public abstract void setEdge(int fromVertex, int toVertex, double weight);

   /**
    * Meldet, ob der Graph die angegebene Kante enthlt.
    *
    * @param fromVertex der Knoten, von welchem die Kante ausgehen soll
    * @param toVertex   der Knoten, zu welchem die Kante hinfhren soll
    *
    * @throws IndexOutOfBoundsException falls fromVertex oder toVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public abstract boolean containsEdge(int fromVertex, int toVertex);

   /**
    * Liefert das Kantengewicht einer Kante.
    * Existiert die angegebene Kante nicht, so wirft die Methode
    * eine IllegalStateException.
    *
    * @param fromVertex der Knoten, von welchem die Kante ausgeht
    * @param toVertex   der Knoten, zu welchem die Kante hinfhrt
    *
    * @throws IndexOutOfBoundsException falls fromVertex oder toVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    * @throws IllegalStateException     falls die angegebene Kante nicht existiert.
    */
   public abstract double getEdgeWeight(int fromVertex, int toVertex);

   /**
    * Entfernt die angegebene Kante aus dem Graphen.
    *
    * @param fromVertex der Knoten, von welchem die Kante ausgeht
    * @param toVertex   der Knoten, zu welchem die Kante hinfhrt
    *
    * @return true falls die Kante vorher enthalten war.
    *
    * @throws IndexOutOfBoundsException falls fromVertex oder toVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public abstract boolean removeEdge(int fromVertex, int toVertex);

   /**
    * Die innere Klasse EdgeIterator stellt einen Iterator ber alle
    * Kanten dieses Graphen dar.
    *
    * Diese Implementierung fgt einfach die von den einzelnen Iteratoren
    * zu den einzelnen Knoten zurckgelieferten Werte aneinander.
    *
    * @author Clemens Bruckmann
    *
    * @version 2005-05-25
    */
   protected class EdgeIterator implements java.util.Iterator<Graph.Edge> {

      /**
       * Der Knoten, dessen Iterator gerade herangezogen wird.
       */
      protected int currentVertex;

      /**
       * Der Iterator, dessen Kanten gerade geliefert werden.
       */
      protected java.util.Iterator<Graph.Edge> currentIterator;

      /**
       * Die nchste zu liefernde Kante.
       */
      protected Graph.Edge nextEdge = null;

      /**
       * Konstruktor.
       */
      protected EdgeIterator() {
         currentVertex = 0;
         currentIterator = null;
      }

      /**
       * Liefert die nchste Kante.
       *
       * @throws java.util.NoSuchElementException falls es keine weitere
       *                                          Kante gibt.
       */
      public Graph.Edge next() {
         if (nextEdge != null) {
            // in diesem Fall wissen wir schon, welches die nchste
            // zu liefernde Kante ist (das hat die Methode hasNext
            // schon herausgefunden) -> wir liefern diese Kante
            Graph.Edge retval = nextEdge;
            nextEdge = null;
            return retval;
         }
         // fr jeden Knoten im Graphen ...
         while (currentVertex < getNumberOfVertices()) {
            // ... gehen wir die Adjazenzliste durch
            if (currentIterator == null) {
               currentIterator = edgeIterator(currentVertex);
            }
            if (currentIterator.hasNext()) {
               // Kante gefunden
               Graph.Edge e = currentIterator.next();
               // bei einem ungerichteten Graphen wird jede Kante doppelt
               // eingetragen, wir geben sie aber nur ein Mal zurck
               if (directed() || e.getFromVertex() <= e.getToVertex()) {
                  return e;
               }
            } else {
               // die Adjazenzliste dieses Knotens ist erschpft,
               // wir gehen zum nchsten Knoten
               ++currentVertex;
               currentIterator = null;
            }
         }
         // wir sind alle Knoten durchgegangen
         throw new java.util.NoSuchElementException();
      }

      /**
       * Liefert true falls der Iterator noch weitere Kanten bereithlt.
       */
      public boolean hasNext() {
         // wissen wir die nchste Kante schon?
         if (nextEdge != null) return true;
         // wir suchen die nchste Kante
         // dabei gehen wir genau wie in der Methode next vor
         while (currentVertex < getNumberOfVertices()) {
            if (currentIterator == null) {
               currentIterator = edgeIterator(currentVertex);
            }
            if (currentIterator.hasNext()) {
               Graph.Edge e = currentIterator.next();
               if (directed() || e.getFromVertex() <= e.getToVertex()) {
                  // nchste zu liefernde Kante gefunden, wir speichern sie
                  nextEdge = e;
                  return true;
               }
            } else {
               // die Adjazenzliste dieses Knotens ist erschpft,
               // wir gehen zum nchsten Knoten
               ++currentVertex;
               currentIterator = null;
            }
         }
         // wir sind alle Knoten durchgegangen, es gibt keine weiteren Kanten
         return false;
      }

      public void remove() {
         throw new UnsupportedOperationException();
      }
   }

   /**
    * Liefert einen Iterator ber alle Kanten dieses Graphen.
    * Die Reihenfolge, in welcher diese Kanten zu liefern sind,
    * ist nicht allgemein nher spezifiziert.
    */
   public java.util.Iterator<Graph.Edge> edgeIterator() {
      return new EdgeIterator();
   }

   /**
    * Liefert einen Iterator ber alle Kanten, die vom angegebenen
    * Knoten ausgehen.
    * Die Reihenfolge, in welcher diese Kanten zu liefern sind,
    * ist nicht allgemein nher spezifiziert.
    *
    * @param fromVertex der Knoten, dessen ausgehende Kanten
    *                   der Iterator liefern soll.
    *
    * @throws IndexOutOfBoundsException falls fromVertex auerhalb des zulssigen Bereichs liegt
    *                                   (0 .. getNumberOfVertices()-1).
    */
   public abstract java.util.Iterator<Graph.Edge> edgeIterator(int fromVertex);


   // Weitere Methoden

   /**
    * Liefert eine einfache textuelle Reprsentation dieses Graphen.
    * Diese Darstellung enthlt die Knotenmenge und die Kantenmenge.
    */
   public String toString() {
      StringBuilder sb = new StringBuilder();
      sb.append(this.getClass().getName()).append(" (")
        .append(this.directed() ? "gerichtet" : "ungerichtet")
        .append("): ({");
      boolean firstElement = true;
      for (int i = 0; i < this.getNumberOfVertices(); ++i) {
         if (!firstElement) {
            sb.append(", ");
         }
         String vertexLabel = this.getVertexLabel(i);
         sb.append(((vertexLabel == null) ? "<null>" : vertexLabel));
         firstElement = false;
      }
      sb.append("}, {");
      for(java.util.Iterator<Graph.Edge> it = this.edgeIterator(); it.hasNext(); ) {
         Graph.Edge edge = it.next();
         String fromVertexLabel = this.getVertexLabel(edge.getFromVertex());
         String toVertexLabel   = this.getVertexLabel(edge.getToVertex());
         sb.append("(")
           .append(((fromVertexLabel == null) ? "<null>" : fromVertexLabel))
           .append(" -> ")
           .append(((toVertexLabel == null) ? "<null>" : toVertexLabel))
           .append(": ")
           .append(edge.getWeight())
           .append(")");
         if (it.hasNext()) {
            sb.append(", ");
         }
      }
      sb.append("})");
      return sb.toString();
   }

   // Methode toXgmml()
   public String toXgmml() {
     StringBuffer xmlSb = new StringBuffer();
     final String NEWLINE = System.getProperty("line.separator");
     xmlSb.append("<?xml version=\"1.0\" encoding=\"ISO-8859-15\"?>").append(NEWLINE)
          .append("<!DOCTYPE graph").append(NEWLINE)
          .append("          PUBLIC \"-//John Punin//DTD graph description//EN\"").append(NEWLINE)
          .append("          \"http://www.cs.rpi.edu/~puninj/XGMML/xgmml.dtd\">").append(NEWLINE)
          .append("<graph xmlns=\"http://www.cs.rpi.edu/XGMML\" directed=\"" + (this.directed() ? 1 : 0) + "\" Vendor=\" WhoamI?\">").append(NEWLINE);
     for (int i = 0; i < this.getNumberOfVertices(); ++i) {
        xmlSb.append("<node id=\"" + i + "\" label=\"" + this.getVertexLabel(i) + "\"/>").append(NEWLINE);
     }
     for(java.util.Iterator<Graph.Edge> it = this.edgeIterator(); it.hasNext(); ) {
       Graph.Edge edge = it.next();
       xmlSb.append("<edge source=\"" + edge.getFromVertex() + "\" target=\"" + edge.getToVertex() + "\"");
       if (edge.getWeight() > 0) {
         xmlSb.append(" weight=\"" + (int)edge.getWeight() + "\"");
       }
       xmlSb.append("/>").append(NEWLINE);
     }
     xmlSb.append("</graph>").append(NEWLINE);
     return xmlSb.toString();
   }

public void printDfsAnalyse(int startVertex, int zielVertex) {
	boolean[] visited = new boolean[this.getNumberOfVertices()];
	java.util.Stack<java.util.Iterator<Graph.Edge>> s = new java.util.Stack<java.util.Iterator<Graph.Edge>>();
	java.util.Stack index = new java.util.Stack();
	java.util.Stack ausgabe=new java.util.Stack();


				visited[startVertex] = true;
				s.push(this.edgeIterator(startVertex));
				index.push(startVertex);
					do {
						if (s.peek().hasNext()) {
							Graph.Edge currentEdge = s.peek().next();
							int toVertex = currentEdge.getToVertex();
							if (toVertex==zielVertex)
							{
								

									while(!index.isEmpty())
									{
										ausgabe.push(index.pop());
									}
								       break;	
								

								
							}
								if (!visited[toVertex]) {

									visited[toVertex] = true;
									s.push(this.edgeIterator(toVertex));
									index.push(toVertex);
								} else {
								
								}
								} else {
								s.pop();
								index.pop();
								}
								} while(!s.isEmpty());
					if(ausgabe.empty())
					{
						System.out.println("Leider nicht gefunden!");
					}
					else
					{
						while(!ausgabe.empty())
						{
						System.out.print(ausgabe.pop()+"->");
						}
						System.out.println(zielVertex);
					}
					
							
				
	}


	public double[] ssspp(int startVertex) //single source shortest parent paths
	{
		double[] distance = new double[this.getNumberOfVertices()];
		boolean[] visited = new boolean[this.getNumberOfVertices()];
		int[] parent=new int[this.getNumberOfVertices()];
		visited[startVertex] = true;
		for (int i = 0; i < distance.length; ++i) parent[i] = -1;
		parent[startVertex]=-2;
		for (int i = 0; i < distance.length; ++i) distance[i] = Double.POSITIVE_INFINITY;
		for (java.util.Iterator<Graph.Edge> it=this.edgeIterator(startVertex);it.hasNext();) 
		{
			Graph.Edge edge = it.next();
			distance[edge.getToVertex()] = edge.getWeight();
			parent[edge.getToVertex()]=0;
		}
		distance[startVertex] = 0;
		for (int wh = 0; wh < this.getNumberOfVertices() - 1; ++wh) 
		{
			double minWeight = Double.POSITIVE_INFINITY;
			int minVertex = -1;
			for (int i = 0; i < this.getNumberOfVertices(); i++) 
			{
				if ((!visited[i]) && minWeight >= distance[i]) 
				{
					minWeight = distance[i];
					minVertex = i;
				}
			}
			if (distance[minVertex] == Double.POSITIVE_INFINITY) return distance;
			visited[minVertex] = true;
			for(java.util.Iterator<Graph.Edge> it=this.edgeIterator(minVertex);it.hasNext();) 
			{
				Graph.Edge edge = it.next();
				if (distance[edge.getToVertex()] > distance[minVertex] + edge.getWeight()) 
					{
						distance[edge.getToVertex()] = distance[minVertex] + edge.getWeight();
						parent[edge.getToVertex()]=minVertex;

					}

			}

		}
		double hilfsfeld[]=new double[this.getNumberOfVertices()*2];
		
		
		
		for(int i=0;i < distance.length;++i)
		{
			hilfsfeld[i]=distance[i];
			hilfsfeld[i+distance.length]=parent[i];
		}


		return hilfsfeld;

	}
}
