package at.ac.tuwien.dslab.rmi.server.test;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;

import junit.framework.TestCase;
import at.ac.tuwien.dslab.rmi.common.impl.DocumentAlreadyExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.DocumentImpl;
import at.ac.tuwien.dslab.rmi.common.impl.DocumentNotExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.FileManException;
import at.ac.tuwien.dslab.rmi.common.impl.RepositoryExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.RepositoryNotExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.UserAccessException;
import at.ac.tuwien.dslab.rmi.common.impl.UserImpl;
import at.ac.tuwien.dslab.rmi.common.interfaces.IDocument;
import at.ac.tuwien.dslab.rmi.common.interfaces.IRepository;
import at.ac.tuwien.dslab.rmi.server.RepositoryImpl;

/**
 * <code>RepositoryImplTest</code> contains a bunch of JUnit tests for the
 * class <code>RepositoryImpl</code> 
 *
 * @author Paul Staroch
 */
public class RepositoryImplTest extends TestCase {
	/**
	 * Tests simple creation and deletion of a repository.
	 * @throws Exception
	 */
	public void testCreateDelete() throws Exception {
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		repository.create();
		assertTrue((new File("repo/bla")).exists());
		assertTrue((new File("repo/bla/.config")).exists());
		repository.delete(new UserImpl("a","b"));
		assertTrue(!(new File("repo/bla")).delete());
		assertTrue(!(new File("repo/bla")).exists());
		assertTrue((new File("repo")).delete());
	}
	
	/**
	 * Tests creation of repository in a repository root that does not exist
	 * @throws Exception
	 */
	public void testRepoRootDoesNotExist() throws Exception {
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		try {
			repository.create();
			fail("IOException not thrown.");
		}
		catch (IOException e) {
			/* do nothing */
		}
	}
	
	/**
	 * Tests creation of repository without owner
	 * @throws Exception
	 */
	public void testCreateRepoNoOwner() throws Exception {
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		try {
			repository.create();
			fail("IOException not thrown.");
		}
		catch (IOException e) {
			/* do nothing */
		}
		assertTrue((new File("repo")).delete());
	}

	/**
	 * Tests creation of a repository without name.
	 * @throws Exception
	 */
	public void testCreateRepoNoName() throws Exception {
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setOwner(new UserImpl("a","b"));
		try {
			repository.create();
			fail("IllegalArgumentException not thrown.");
		}
		catch (IllegalArgumentException e) {
			/* do nothing */
		}
		assertTrue((new File("repo")).delete());
	}
	
	/**
	 * Copies <code>file1</code> to <code>file2</code>.
	 * @param file1 Source file
	 * @param file2 Destination file
	 * @throws IOException
	 */
	private void copyFile(String file1, String file2) throws IOException {
		FileInputStream in=new FileInputStream(file1);
		FileOutputStream out=new FileOutputStream(file2);
		
		int data;
		while((data=in.read())!=-1) {
			out.write(data);
		}
		
		in.close();
		out.close();
	}
	
	/**
	 * Tests creation of a repository with a repository root that already
	 * exists, but is not a directory.
	 * @throws Exception
	 */
	public void testCreateRootFile() throws Exception {
		copyFile("build.xml","repo");
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		try {
			repository.create();
			fail("IOException not thrown.");
		}
		catch (IOException e) {
			/* do nothing */
		}
		assertTrue((new File("repo")).delete());
	}
	
	/**
	 * Retrieves the content of a file as <code>byte[]</code>.
	 * @param filename name of the file to be read
	 * @return content of file <code>filename</code>
	 * @throws IOException if an error occurs while reading
	 * <code>filename</code>.
	 */
	private byte[] getFileContent(String filename) throws IOException {
		File file=new File(filename);
		
		byte[] data=new byte[(int)file.length()];
		FileInputStream inputStream=new FileInputStream(file);
		inputStream.read(data);
		inputStream.close();
		
		return data;
	}
	
	/**
	 * Tests creating a document without creating the repository
	 * @throws Exception
	 */
	public void testCreateDocumentWithoutRepository() throws Exception {
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		IDocument document=
			new DocumentImpl("build.xml",1,getFileContent("build.xml"));
		try {
			repository.createDocument(document);
			fail("IOException not thrown.");
		}
		catch (IOException e) {
			/* do nothing */
		}
		assertTrue((new File("repo")).delete());
	}

	/**
	 * Tests creating a document.
	 * @throws Exception
	 */
	public void testCreateDocument() throws Exception {
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		repository.create();
		IDocument document=
			new DocumentImpl("build.xml",1,getFileContent("build.xml"));
		repository.createDocument(document);
		repository.delete(new UserImpl("a","b"));
		assertTrue((new File("repo")).delete());
	}
	
	/**
	 * Tests creating multiple documents.
	 * @throws Exception
	 */
	public void testCreateMultipleDocument() throws Exception {
		copyFile("build.xml","test1");
		copyFile("build.xml","test2");
		copyFile("build.xml","test3");
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		repository.create();
		repository.createDocument((new DocumentImpl
				("build.xml",1,getFileContent("build.xml"))));
		repository.createDocument((new DocumentImpl
				("test1",1,getFileContent("test1"))));
		repository.createDocument((new DocumentImpl
				("test2",1,getFileContent("test2"))));
		repository.createDocument((new DocumentImpl
				("test3",1,getFileContent("test3"))));
		repository.delete(new UserImpl("a","b"));
		assertTrue((new File("repo")).delete());
		assertTrue((new File("test1")).delete());
		assertTrue((new File("test2")).delete());
		assertTrue((new File("test3")).delete());
	}

	/**
	 * Tests creating a document that already exists
	 * @throws Exception
	 */
	public void testCreateDocumentAlreadyExists() throws Exception {
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		repository.create();
		IDocument document=
			new DocumentImpl("build.xml",1,getFileContent("build.xml"));
		repository.createDocument(document);
		try {
			repository.createDocument(document);
			fail("IOException not thrown.");
		}
		catch (DocumentAlreadyExistsException e) {
			/* do nothing */
		}
		repository.delete(new UserImpl("a","b"));
		assertTrue((new File("repo")).delete());
	}

	/**
	 * Tests creating a document with a version > 1.
	 * @throws Exception
	 */
	public void testCreateDocumentWrongVersion() throws Exception {
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		repository.create();
		IDocument document=
			new DocumentImpl("build.xml",2,getFileContent("build.xml"));
		try {
			repository.createDocument(document);
			fail("IllegalArgumentException not thrown.");
		}
		catch (IllegalArgumentException e) {
			/* do nothing */
		}
		repository.delete(new UserImpl("a","b"));
		assertTrue((new File("repo")).delete());
	}
	
	/**
	 * Tests deleting a repository that has not been created
	 * @throws Exception
	 */
	public void testDeleteNoCreate() throws Exception {
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		
		try {
			repository.delete(new UserImpl("a","b"));
			fail("RepositoryNotExists not thrown.");
		}
		catch (RepositoryNotExistsException e) {
			/* do nothing */
		}
		assertTrue((new File("repo")).delete());
	}

	/**
	 * Compares two byte arrays.
	 * @param array1
	 * @param array2
	 * @return <code>true</code>, if and only if <code>array1</code> equals
	 * <code>array2</code>, <code>false</code> otherwise
	 */
	private boolean compareByteArray(byte[] array1, byte[] array2) {
		if(array1.length!=array2.length) {
			return false;
		}
		
		for(int a=0;a<array1.length;a++) {
			if(array1[a]!=array2[a]) {
				return false;
			}
		}
		
		return true;
	}
	
	/**
	 * Tests <code>getAllDocuments</code>
	 * @throws Exception
	 */
	public void testGetAllDocuments() throws Exception {
		copyFile("build.xml","test1");
		copyFile("build.xml","test2");
		copyFile("build.xml","test3");
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		repository.create();
		repository.createDocument
			((new DocumentImpl("build.xml",1,getFileContent("build.xml"))));
		repository.createDocument
			((new DocumentImpl("test1",1,getFileContent("test1"))));
		repository.createDocument
			((new DocumentImpl("test2",1,getFileContent("test2"))));
		repository.createDocument
			((new DocumentImpl("test3",1,getFileContent("test3"))));
		IDocument[] documents=repository.getAllDocuments();
		assertTrue(documents[0].getName().equals("build.xml"));
		assertTrue(documents[1].getName().equals("test1"));
		assertTrue(documents[2].getName().equals("test2"));
		assertTrue(documents[3].getName().equals("test3"));
		assertTrue(compareByteArray
				(documents[0].getContent(),getFileContent("build.xml")));
		assertTrue(compareByteArray
				(documents[1].getContent(),getFileContent("test1")));
		assertTrue(compareByteArray
				(documents[2].getContent(),getFileContent("test2")));
		assertTrue(compareByteArray
				(documents[3].getContent(),getFileContent("test3")));
		assertTrue(documents[0].getVersion()==1);
		assertTrue(documents[1].getVersion()==1);
		assertTrue(documents[2].getVersion()==1);
		assertTrue(documents[3].getVersion()==1);
		repository.delete(new UserImpl("a","b"));
		assertTrue((new File("repo")).delete());
		assertTrue((new File("test1")).delete());
		assertTrue((new File("test2")).delete());
		assertTrue((new File("test3")).delete());
	}
	
	/**
	 * Tests <code>getDocument</code>.
	 * @throws Exception
	 */
	public void testGetDocument() throws Exception {
		copyFile("build.xml","test1");
		copyFile("build.xml","test2");
		copyFile("build.xml","test3");
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		repository.create();
		
		ArrayList<String> files=new ArrayList<String>();
		files.add("build.xml");
		files.add("test1");
		files.add("test2");
		files.add("test3");
		
		for(String filename : files) {
			repository.createDocument
				((new DocumentImpl(filename,1,getFileContent(filename))));
		}
		
		for(String filename : files) {
			IDocument document=repository.getDocument(filename, 1);
			assertTrue(document.getName().equals(filename));
			assertTrue(compareByteArray
					(document.getContent(),getFileContent(filename)));
			assertTrue(document.getVersion()==1);
		}
		repository.delete(new UserImpl("a","b"));
		assertTrue((new File("repo")).delete());
		assertTrue((new File("test1")).delete());
		assertTrue((new File("test2")).delete());
		assertTrue((new File("test3")).delete());
	}
	
	/**
	 * Tests if <code>setAllowedUsers</code> and <code>getAllowedUsers</code>
	 * work
	 * @throws Exception
	 */
	public void testAllowedUsers() throws Exception {
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		repository.create();
		String[] allowedUsers={"user1","user2","user3"};
		repository.setAllowedUsers(allowedUsers);
		String[] users=repository.getAllowedUsers();
		for(int a=0;a<3;a++) {
			assertTrue(allowedUsers[a]==users[a]);
		}
		repository.delete(new UserImpl("a","b"));
		assertTrue((new File("repo")).delete());
	}
	
	/**
	 * Tests deleting repository with a user that is not allowed to delete the
	 * repository
	 * @throws Exception
	 */
	public void testDeleteWrongUser() throws Exception {
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		repository.create();
		try {
			repository.delete(new UserImpl("a","c"));
			fail("UserAccessException not thrown.");
		}
		catch (UserAccessException e) {
			/* do nothing */
		}
		try {
			repository.delete(new UserImpl("b","c"));
			fail("UserAccessException not thrown.");
		}
		catch (UserAccessException e) {
			/* do nothing */
		}
		try {
			repository.delete(new UserImpl("b","b"));
			fail("UserAccessException not thrown.");
		}
		catch (UserAccessException e) {
			/* do nothing */
		}
		repository.delete(new UserImpl("a","b"));
		assertTrue((new File("repo")).delete());
	}

	/**
	 * Tests removing a document
	 * @throws Exception
	 */
	public void testRemoveDocument() throws Exception {
		copyFile("build.xml","test1");
		copyFile("build.xml","test2");
		copyFile("build.xml","test3");
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		repository.create();
		
		ArrayList<String> files=new ArrayList<String>();
		files.add("build.xml");
		files.add("test1");
		files.add("test2");
		files.add("test3");
		
		for(String filename : files) {
			repository.createDocument
				((new DocumentImpl(filename,1,getFileContent(filename))));
		}
		
		for(String filename : files) {
			IDocument document=repository.removeDocument(filename);
			assertTrue(document.getName().equals(filename));
			assertTrue(compareByteArray
					(document.getContent(),getFileContent(filename)));
			assertTrue(document.getVersion()==1);
		}
		
		for(String filename : files) {
			assertTrue(repository.getDocument(filename,1)==null);
		}
		
		for(String filename : files) {
			try {
				repository.removeDocument(filename);
				fail("DocumentNotExistsException not thrown.");
			}
			catch(DocumentNotExistsException e) {
				/* do nothing */
			}
		}
		for(String filename : files) {
			repository.createDocument
				((new DocumentImpl(filename,1,getFileContent(filename))));
		}
		
		repository.delete(new UserImpl("a","b"));
		assertTrue((new File("repo")).delete());
		assertTrue((new File("test1")).delete());
		assertTrue((new File("test2")).delete());
		assertTrue((new File("test3")).delete());
	}

	/**
	 * Tests creating, removing and updating documents without creating a
	 * repository
	 * @throws Exception
	 */
	public void testCreateRemoveUpdateDocumentWithoutCreateRepository()
			throws Exception {
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		
		try {
			repository.createDocument((new DocumentImpl
					("build.xml",1,getFileContent("build.xml"))));
			fail("RepositoryNotExistsException not thrown.");
		}
		catch (IOException e) {
			/* do nothing */
		}
	
		try {
			repository.updateDocument((new DocumentImpl
					("build.xml",1,getFileContent("build.xml"))));
			fail("RepositoryNotExistsException not thrown.");
		}
		catch (RepositoryNotExistsException e) {
			/* do nothing */
		}
		
		try {
			repository.removeDocument("build.xml");
			fail("RepositoryNotExistsException not thrown.");
		}
		catch (IOException e) {
			/* do nothing */
		}
		
		try {
			repository.getDocument("build.xml",1);
			fail("RepositoryNotExistsException not thrown.");
		}
		catch (IOException e) {
			/* do nothing */
		}
		
		try {
			repository.getAllDocuments();
			fail("RepositoryNotExistsException not thrown.");
		}
		catch (IOException e) {
			/* do nothing */
		}
		assertTrue((new File("repo")).delete());
	}
	
	/**
	 * Tests updating documents.
	 * @throws Exception
	 */
	public void testUpdateDocument() throws Exception {
		copyFile("build.xml","test1");
		copyFile("build.xml","test2");
		copyFile("build.xml","test3");
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		repository.create();
		
		ArrayList<String> files=new ArrayList<String>();
		files.add("build.xml");
		files.add("test1");
		files.add("test2");
		files.add("test3");
		
		for(String filename : files) {
			repository.createDocument((new DocumentImpl
					(filename,1,getFileContent(filename))));
		}
		
		for(int a=1;a<5;a++) {
			for(String filename : files) {
				repository.updateDocument((new DocumentImpl
						(filename,a,getFileContent(filename))));
			}
		}

		try {
			repository.updateDocument((new DocumentImpl
					("bla",2,getFileContent("build.xml"))));
			fail("DocumentNotExistsException not thrown.");
		}
		catch(DocumentNotExistsException e) {
			/* do nothing */
		}
		
		for(String filename : files) {
			try {
				repository.updateDocument((new DocumentImpl
						(filename,7,getFileContent(filename))));
				fail("FileManException not thrown.");
			}
			catch(FileManException e) {
				/* do nothing */
			}
		}
		
		for(int a=1;a<6;a++) {
			for(String filename : files) {
				IDocument document=repository.getDocument(filename, a);
				assertTrue(document.getName().equals(filename));
				assertTrue(compareByteArray
						(document.getContent(),getFileContent(filename)));
				assertTrue(document.getVersion()==a);
			}
		}
		
		for(String filename : files) {
			repository.removeDocument(filename);
		}
		
		for(int a=1;a<6;a++) {
			for(String filename : files) {
				assertTrue(repository.getDocument(filename, a)==null);
			}
		}
		
		repository.delete(new UserImpl("a","b"));
		assertTrue((new File("repo")).delete());
		assertTrue((new File("test1")).delete());
		assertTrue((new File("test2")).delete());
		assertTrue((new File("test3")).delete());
	}
	
	/**
	 * Tests if an exception is thrown when a repository's name contains
	 * ".." or "/".
	 * @throws Exception
	 */
	public void testNaming() throws Exception {
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		repository.create();

		try {
			repository.createDocument((new DocumentImpl
					("..",1,getFileContent("build.xml"))));
			fail("IllegalArgumentException not thrown.");
		}
		catch (IllegalArgumentException e) {
			/* do nothing */
		}
		
		try {
			repository.createDocument((new DocumentImpl
					("repo"+File.separator+"repo",1,
							getFileContent("build.xml"))));
			fail("IllegalArgumentException not thrown.");
		}
		catch (IllegalArgumentException e) {
			/* do nothing */
		}
		
		try {
			repository.updateDocument((new DocumentImpl
					("..",1,getFileContent("build.xml"))));
			fail("IllegalArgumentException not thrown.");
		}
		catch (IllegalArgumentException e) {
			/* do nothing */
		}
		
		try {
			repository.updateDocument((new DocumentImpl
					("repo"+File.separator+"repo",1,
							getFileContent("build.xml"))));
			fail("IllegalArgumentException not thrown.");
		}
		catch (IllegalArgumentException e) {
			/* do nothing */
		}
		
		try {
			repository.getDocument("..",1);
			fail("IllegalArgumentException not thrown.");
		}
		catch (IllegalArgumentException e) {
			/* do nothing */
		}
		
		try {
			repository.getDocument("repo"+File.separator+"repo",1);
			fail("IllegalArgumentException not thrown.");
		}
		catch (IllegalArgumentException e) {
			/* do nothing */
		}
		
		try {
			repository.removeDocument("..");
			fail("IllegalArgumentException not thrown.");
		}
		catch (IllegalArgumentException e) {
			/* do nothing */
		}
		
		try {
			repository.removeDocument("repo"+File.separator+"repo");
			fail("IllegalArgumentException not thrown.");
		}
		catch (IllegalArgumentException e) {
			/* do nothing */
		}
		
		repository.delete(new UserImpl("a","b"));
		assertTrue((new File("repo")).delete());
	}
	
	/**
	 * Tests if importing a file named ".config" into the repository works
	 * correctly and does not conflict with the repository config file.
	 * @throws Exception
	 */
	public void testCreateConfigDocument() throws Exception {
		assertTrue((new File("repo")).mkdir());
		IRepository repository=new RepositoryImpl("repo");
		repository.setName("bla");
		repository.setOwner(new UserImpl("a","b"));
		repository.create();
		IDocument document=new DocumentImpl(".config",1,
				getFileContent("build.xml"));
		repository.createDocument(document);
		document=repository.getDocument(".config", 1);
		assertTrue(document.getVersion()==1);
		assertTrue(document.getName().equals(".config"));
		assertTrue(compareByteArray(document.getContent(),
				getFileContent("build.xml")));
		repository.delete(new UserImpl("a","b"));
		assertTrue((new File("repo")).delete());
	}
	
	/**
	 * Tests creating a repository that already exists.
	 * @throws Exception
	 */
	public void testCreateExistingRepository() throws Exception {
		assertTrue((new File("repo")).mkdir());
		IRepository repository1=new RepositoryImpl("repo");
		repository1.setName("bla");
		repository1.setOwner(new UserImpl("a","b"));
		repository1.create();
		IRepository repository2=new RepositoryImpl("repo");
		repository2.setName("bla");
		repository2.setOwner(new UserImpl("a","b"));
		try {
			repository2.create();
			fail("RepositoryExistsException not thrown.");
		}
		catch(RepositoryExistsException e) {
			/* do nothing */
		}
		repository1.delete(new UserImpl("a","b"));
		assertTrue((new File("repo")).delete());
	}
	
	/**
	 * Tests the maintainance of multiple repositories.
	 * @throws Exception
	 */
	public void testMultipleRepositories() throws Exception {
		ArrayList<String> repos=new ArrayList<String>();
		for(int a=0;a<10;a++) {
			repos.add("bla"+a);
		}
		copyFile("build.xml","test1");
		copyFile("build.xml","test2");
		copyFile("build.xml","test3");
		assertTrue((new File("repo")).mkdir());
		
		ArrayList<IRepository> repositories=new ArrayList<IRepository>();
		for(String repoName : repos) {
			IRepository repository=new RepositoryImpl("repo");
			repository.setName(repoName);
			repository.setOwner(new UserImpl("a","b"));
			repository.create();
			repositories.add(repository);
			assertTrue((new File("repo/"+repoName)).exists());
			assertTrue((new File("repo/"+repoName+"/.config")).exists());
		}
		
		ArrayList<String> files=new ArrayList<String>();
		files.add("build.xml");
		files.add("test1");
		files.add("test2");
		files.add("test3");
		
		for(IRepository repository : repositories) {
			for(String filename : files) {
				repository.createDocument((new DocumentImpl(
						filename,1,getFileContent(filename))));
			}
		}
		
		for(IRepository repository : repositories) {
			for(int a=1;a<5;a++) {
				for(String filename : files) {
					repository.updateDocument((new DocumentImpl
							(filename,a,getFileContent(filename))));
				}
			}
		}

		for(IRepository repository : repositories) {
			try {
				repository.updateDocument((new DocumentImpl
						("bla",2,getFileContent("build.xml"))));
				fail("DocumentNotExistsException not thrown.");
			}
			catch(DocumentNotExistsException e) {
				/* do nothing */
			}
		}
		
		for(IRepository repository : repositories) {
			for(String filename : files) {
				try {
					repository.updateDocument((new DocumentImpl
							(filename,7,getFileContent(filename))));
					fail("FileManException not thrown.");
				}
				catch(FileManException e) {
					/* do nothing */
				}
			}
		}
		
		for(IRepository repository : repositories) {
			for(int a=1;a<6;a++) {
				for(String filename : files) {
					IDocument document=repository.getDocument(filename, a);
					assertTrue(document.getName().equals(filename));
					assertTrue(compareByteArray
							(document.getContent(),getFileContent(filename)));
					assertTrue(document.getVersion()==a);
				}
			}
			for(String filename : files) {
				IDocument document=repository.getDocument(filename, -1);
				assertTrue(document.getName().equals(filename));
				assertTrue(compareByteArray
						(document.getContent(),getFileContent(filename)));
				assertTrue(document.getVersion()==5);
			}
		}
		
		for(IRepository repository : repositories) {
			for(String filename : files) {
				repository.removeDocument(filename);
			}
		}
		
		for(IRepository repository : repositories) {
			for(int a=1;a<6;a++) {
				for(String filename : files) {
					assertTrue(repository.getDocument(filename, a)==null);
				}
			}
		}
		
		for(IRepository repository : repositories) {
			String repoName=repository.getName();
			repository.delete(new UserImpl("a","b"));
			assertTrue(!(new File("repo/"+repoName)).exists());
		}
		
		assertTrue((new File("repo")).delete());
		assertTrue((new File("test1")).delete());
		assertTrue((new File("test2")).delete());
		assertTrue((new File("test3")).delete());
	}
}
