/*
 * Distributed Systems
 * Winter Term 2006/07
 * Lab 3
 * Paul Staroch, 0425426
 */

package at.ac.tuwien.dslab.rmi.client;

import at.ac.tuwien.dslab.rmi.common.interfaces.IRepositoryListener;
import at.ac.tuwien.dslab.rmi.common.interfaces.IDocument;
import at.ac.tuwien.dslab.rmi.common.interfaces.IFileManager;
import at.ac.tuwien.dslab.rmi.common.impl.UserImpl;
import at.ac.tuwien.dslab.rmi.common.impl.RepositoryNotExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.UserAccessException;
import at.ac.tuwien.dslab.rmi.common.impl.FileManException;

import java.util.Scanner;
import java.io.Serializable;

import java.rmi.NotBoundException;
import java.rmi.RemoteException;
import java.rmi.Naming;
import java.rmi.server.UnicastRemoteObject;

import java.net.MalformedURLException;

/**
 * RepositoryMonitor implements a simple monitor that registers itself at the
 * repository server and is notified when any file is imported, committed or
 * removed.
 */
public class RepositoryMonitor
		extends UnicastRemoteObject
		implements IRepositoryListener, Serializable {
	/**
	 * Default constructor
	 *
	 * @throws RemoteException if an error occurs while exporting this
	 * object
	 */
	protected RepositoryMonitor() throws RemoteException {
		super();
	}

	/**
	 * cancels the exportation of this object (necessary for a clean
	 * shutdown of the monitor)
	 */
	protected void unexport() {
		try {
			unexportObject(this, true);
		}
		catch (Exception e) {
			/* do nothing */
		}
	}
	
	/**
	 * being called by the server when a document is being committed
	 *
	 * @param doc document that has been committed
	 */
	public void documentCommitted(IDocument doc) {
		System.out.println("A document has been committed: ");
		showDocumentInformation(doc);
	}

	/**
	 * being called by the server when a document is being imported
	 *
	 * @param doc document that has been imported
	 */
	public void documentImported(IDocument doc) {
		System.out.println("A document has been imported: ");
		showDocumentInformation(doc);
	}

	/**
	 * being called by the server when a document is being deleted
	 *
	 * @param doc document that has been deleted
	 */
	public void documentRemoved(IDocument doc) {
		System.out.println("A document has been removed: ");
		showDocumentInformation(doc);
	}

	/**
	 * shows name, version and length of a specified document
	 *
	 * @param doc document to show the information about
	 */
	private void showDocumentInformation(IDocument doc) {
		System.out.println(" Name: "+doc.getName());
		System.out.println(" Version: "+doc.getVersion());
		System.out.println(" Length: "+doc.getContent().length);
		System.out.println();
	}

	/**
	 * Main method.
	 *
	 * An instance <code>RepositoryMonitor</code> will be exported, a 
	 * connection to the RMI server will be established, the exported
	 * instance of <code>RepositoryMonitor</code> will be registered
	 * at the server.
	 *
	 * Afterwards, the method waits for the user to type "exit"; when
	 * that happens, the listener will be deregistered at the server
	 * and unexported before the program terminates.
	 */
	public static void main(String[] args) {
		/*
		 * Arguments:
		 * 1. RMI URL of the repository server
		 * 2. repository name on the server
		 * 3. username for the access to the repository
		 * 4. password for the access to the repository
		 */
		if(args.length!=4) {
			System.err.println
				("Invalid number of command line arguments.");
			return;
		}

		/* initialize listener */
		RepositoryMonitor listener;
		try {
			listener=new RepositoryMonitor();
		}
		catch (RemoteException re) {
			System.err.println("Communication error.");
			return;
		}

		/* initialize access to the server */
		IFileManager fileMan;
		try {
			fileMan=(IFileManager)Naming.lookup(args[0]);
		}
		catch (NotBoundException nbe) {
			System.err.println
				("Name "+args[0]+" is currently not bound.");
			/*
			 * we have to unexport the listener for the sake of
			 * a clean shutdown
			 */
			listener.unexport();
			return;
		}
		catch (MalformedURLException mue) {
			System.err.println("Malformed URL.");
			listener.unexport();
			return;
		}
		catch (RemoteException re) {
			System.err.println("Cannot contact RMI registry.");
			listener.unexport();
			return;
		}

		UserImpl user=new UserImpl(args[2],args[3]);
	
		/* register listener */
		try {
			fileMan.addRepositoryListener(args[1], user, listener);
		}
		catch (RepositoryNotExistsException rnee) {
			System.err.println
				("Specified repository does not exist.");
			listener.unexport();
			return;
		}
		catch (UserAccessException uae) {
			System.err.println("Permission denied.");
			listener.unexport();
			return;
		}
		catch (FileManException fme) {
			System.err.println
				("An error occurred while adding the "+
				 "repository listener.");
			listener.unexport();
			return;
		}
		catch (RemoteException re) {
			System.err.println("Communication error.");
			listener.unexport();
			return;
		}

		/* wait for the user typing "exit" (case-insensitive) */
		Scanner sc=new Scanner(System.in);
		String input="";
		while(!(input=sc.next()).equalsIgnoreCase("exit"));

		/* tidy up */
		try {
			fileMan.removeRespositoryListener(args[1], listener);
		}
		catch (RepositoryNotExistsException rnee) {
			/* may not happen */
		}
		catch (FileManException fme) {
			System.err.println
				("An error occurred while removing the "+
				 "repository listener.");
		}
		catch (RemoteException re) {
			System.err.println("Communication error.");
		}
		
		listener.unexport();
	}
}

