package at.ac.tuwien.dslab.rmi.server;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;
import java.util.TreeSet;

import at.ac.tuwien.dslab.rmi.common.impl.DocumentAlreadyExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.DocumentImpl;
import at.ac.tuwien.dslab.rmi.common.impl.DocumentNotExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.FileManException;
import at.ac.tuwien.dslab.rmi.common.impl.RepositoryExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.RepositoryNotExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.UserAccessException;
import at.ac.tuwien.dslab.rmi.common.impl.UserImpl;
import at.ac.tuwien.dslab.rmi.common.interfaces.IDocument;
import at.ac.tuwien.dslab.rmi.common.interfaces.IRepository;
import at.ac.tuwien.dslab.rmi.common.interfaces.IUser;

public class RepositoryImpl implements IRepository {

	/**
	 * generated serial version UID for this class
	 */
	private static final long serialVersionUID = -1783091794418710814L;
	
	/**
	 * name of the repository encapsulated by this object
	 */
	private String repositoryName;
	
	/**
	 * repository root
	 */
	private String repoRoot;
	
	/**
	 * RMI server URL
	 */
	private String serverUrl;
	
	/**
	 * owner of this repository
	 */
	private IUser owner;
	
	/**
	 * names of the users that are allowed to access the repository
	 */
	private String[] allowedUsers={};
	
	/**
	 * Config file option for the username of the repository owner
	 */
	private static final String CONFIG_OWNER_USERNAME="ownerUsername";
	
	/**
	 * Config file option for the password of the repository owner
	 */
	private static final String CONFIG_OWNER_PASSWORD="ownerPassword";
	
	/**
	 * Config file option for the number of users allowed to access the
	 * repository (except the owner)
	 */
	private static final String CONFIG_USER_COUNT="allowedUsersCount";
	
	/**
	 * Config file option for the username of a user allowed to access the
	 * repository (except the owner)
	 */
	private static final String CONFIG_ALLOWED_USER="allowedUser";
	
	/**
	 * Name of the config file
	 */
	public static final String CONFIG_FILENAME=".config";
	
	/**
	 * Checks if the filename is valid.
	 * A filename is valid if it does not contain ".." or the system dependent
	 * file separator (i.e. "/" on UNIX-like systems or "\" on Windows)
	 * @param filename filename to check
	 * @throws IllegalArgumentException if the filename is invalid
	 */
	private void validFilename(String filename)
			throws IllegalArgumentException {
		if((filename.indexOf("..")>-1) ||
				(filename.indexOf(File.separator)>-1)) {
			throw new IllegalArgumentException
				("A filename may not contain \"..\" or \"/\".");
		}
	}
	
	/**
	 * Constructor
	 * @param repoRoot root directory of the repisitory
	 */
	public RepositoryImpl(String repoRoot) {
		this.repoRoot=repoRoot;
	}

	/**
	 * Checks if the repository root is accessible
	 * @throws IOException if the repository root is not accessible
	 */
	private void checkRepositoryRoot() throws IOException {
		if(repositoryName==null) {
			throw new IllegalArgumentException("Repository root not set");
		}
		
		File file=new File(repoRoot);
		if(!file.exists()) {
			throw new IOException("Repository root does not exist.");
		}
		if(!file.isDirectory()) {
			throw new IOException("Repository root is not a directory.");
		}
		if(!file.canRead()) {
			throw new IOException("Can't read from repository root.");
		}
		if(!file.canWrite()) {
			throw new IOException("Can't write to repository root.");
		}
	}
	
	/**
	 * Creates the Repository
	 * @throws IOException if the repository is not accessible
	 * @throws RepositoryExistsException if the repository already exists
	 */
	public synchronized void create() throws IOException,
			RepositoryExistsException {
		/* check if repository root is accessible */
		checkRepositoryRoot();
		
		if(this.owner==null) {
			throw new IOException("No owner specified.");
		}
		
		/* check if directory exists */
		File file=new File(repoRoot+File.separator+repositoryName);
		if(file.exists()) {
			throw new RepositoryExistsException(repositoryName);
		}
		
		file.mkdir();
		
		/* create configuration */
		Properties config=new Properties();
		config.setProperty(CONFIG_OWNER_USERNAME, owner.getName());
		config.setProperty("ownerPassword", owner.getPassword());
		
		config.setProperty
			(CONFIG_USER_COUNT, Integer.toString(allowedUsers.length));
		for(int a=0;a<allowedUsers.length;a++) {
			config.setProperty(CONFIG_ALLOWED_USER+a,allowedUsers[a]);
		}
		
		FileOutputStream stream=new FileOutputStream
			(repoRoot+File.separator+repositoryName+File.separator+
					CONFIG_FILENAME);
		config.store(stream,"");
		stream.close();
	}

	/**
	 * Creates a new document in the repository
	 * @param document document that shall be added to the repository
	 * @throws IOException if the document can't be created
	 * @throws DocumentAlreadyExistsException if a document with the name
	 * of the document <code>document</code> already exists in the repository
	 */
	public synchronized void createDocument(IDocument document)
			throws IOException, DocumentAlreadyExistsException {
		try {
			load();
		}
		catch (RepositoryNotExistsException e) {
			throw new IOException(e.getMessage());
		}
		
		validFilename(document.getName());
		if(document.getVersion()!=1) {
			throw new IllegalArgumentException("Wrong version number");
		}
		
		if((new File(repoRoot+File.separator+repositoryName+
				File.separator+document.getName()+".1")).exists()) {
			throw new DocumentAlreadyExistsException(document.getName());
		}
		
		FileOutputStream output=new FileOutputStream
			(repoRoot+File.separator+repositoryName+
					File.separator+document.getName()+".1");
		output.write(document.getContent());
		output.flush();
		output.close();
	}

	/**
	 * Deletes the repository
	 * @param user the user that wants to delete the repository
	 * @throws IOException if the repository can't be deleted
	 * @throws UserAccessException when the user <code>user</code> is not
	 * allowed to delete this repository
	 * @throws RepositoryNotExistsException if the repository does not exist
	 */
	public synchronized void delete(IUser user) throws IOException,
			UserAccessException, RepositoryNotExistsException {
		load();
		
		if(!user.equals(owner)) {
			throw new UserAccessException("Permission denied.");
		}
		
		boolean success=true;
		
		File file=new File(repoRoot+File.separator+repositoryName);
		for(String filename : file.list()) {
			File toBeDeleted=new File(repoRoot+File.separator+
					repositoryName+File.separator+filename);
			if(!success) System.out.println(repoRoot+File.separator+
					repositoryName+File.separator+filename);
			success=toBeDeleted.delete() && success;
		}
		success=file.delete() && success;
		if(!success) {
			throw new IOException("Could not remove repository.");
		}
	}

	/**
	 * Strips the version number from a filename
	 * @param filename name of the file where to strip the version number from
	 * @return filename without version number
	 */
	private String stripVersion(String filename) {
		return filename.substring(0,filename.lastIndexOf('.'));
	}
	
	/**
	 * Retrieves all documents (always the newest version) from the repository
	 * @return array with all documents in the repository
	 * @throws IOException if any of the files in the repository is not
	 * accessible
	 */
	public synchronized IDocument[] getAllDocuments() throws IOException {
		try {
			load();
		}
		catch (RepositoryNotExistsException e) {
			throw new IOException(e.getMessage());
		}
		
		TreeSet<String> filenames=new TreeSet<String>();
		
		for(String filename : new File(repoRoot+File.separator+
				repositoryName).list()) {
			if(!filename.equals(CONFIG_FILENAME)) {
				if(!filenames.contains(stripVersion(filename))) {
					filenames.add(stripVersion(filename));
				}
			}
		}
		
		IDocument[] documents=new IDocument[filenames.size()];
		int a=0;
		for(String filename : filenames) {
			documents[a]=getDocument(filename, getHighestVersion(filename));
			a++;
		}
		
		return documents;
	}

	/**
	 * Returns the names of all users that are allowed to access the
	 * repository (except the owner himself)
	 * @return array with the names of all users
	 */
	public String[] getAllowedUsers() {
		return allowedUsers;
	}

	/**
	 * Retrieves a single document from the repository
	 * @param name name of the document to fetch
	 * @param version version of the document to fetch
	 * @return document, if it can be found; <code>null</code> if there is no
	 * document named <code>name</code> with the version <code>version</code>.
	 * @throws IOException if the file cannot be accessed
	 */
	public synchronized IDocument getDocument(String name, long version)
			throws IOException {
		try {
			load();
		}
		catch (RepositoryNotExistsException e) {
			throw new IOException(e.getMessage());
		}
		
		validFilename(name);
		
		if(version==-1) {
			version=getHighestVersion(name);
		}
		
		File file=new File(repoRoot+File.separator+repositoryName+
				File.separator+name+"."+Long.toString(version));
		if(!file.exists()) {
			return null;
		}
		
		if(!file.isFile()) {
			throw new IOException("Can't read from a directory.");
		}
		
		byte[] data=new byte[(int)file.length()];
		FileInputStream inputStream=new FileInputStream(file);
		inputStream.read(data);
		inputStream.close();
		
		return new DocumentImpl(name,version,data);
	}

	/**
	 * Returns the name of the repository.
	 * @return name of the repository
	 */
	public String getName() {
		return repositoryName;
	}

	/**
	 * Returns the owner of the repository.
	 * @return owner of the repository
	 */
	public IUser getOwner() {
		return owner;
	}

	/**
	 * Returns the URL of the RMI server
	 * @return server URL
	 */
	public String getServerUrl() {
		return serverUrl;
	}

	/**
	 * Loads the repository from disk.
	 * @throws IOException if the repository is not accessible
	 * @throws RepositoryNotExistsException if the repository does not exist
	 */
	public synchronized void load() throws IOException,
			RepositoryNotExistsException {
		/* check if repository root is accessible */
		checkRepositoryRoot();

		Properties config=new Properties();
		FileInputStream stream;
		try {
			stream=new FileInputStream(repoRoot+File.separator+repositoryName+
					File.separator+CONFIG_FILENAME);
		}
		catch (FileNotFoundException fnfe) {
			throw new RepositoryNotExistsException(repositoryName);
		}
		
		config.load(stream);
		stream.close();
		owner=new UserImpl();
		owner.setName(config.getProperty(CONFIG_OWNER_USERNAME));
		owner.setPasswordHashed(config.getProperty(CONFIG_OWNER_PASSWORD));
		
		int allowedUserCount=Integer.parseInt
			(config.getProperty(CONFIG_USER_COUNT));
		allowedUsers=new String[allowedUserCount];
		for(int a=0;a<allowedUserCount;a++) {
			allowedUsers[a]=config.getProperty
				(CONFIG_ALLOWED_USER+Integer.toString(a));
		}
	}

	/**
	 * Gets the highest version of a document
	 * @param document name of the document
	 * @return highest version of the document named <code>documenty</code>
	 */
	private long getHighestVersion(String document) {
		long a=1;
		while((new File(repoRoot+File.separator+repositoryName+File.separator+
				document+"."+Long.toString(a))).exists()) {
			a++;
		}
		return a-1;
	}
	
	/**
	 * Permanently removes a document from the repository
	 * @param document name of the document to be removed
	 * @return document that has been removed
	 * @throws IOException if the document cannot be removed
	 * @throws DocumentNotExistsException if there is no document named
	 * <code>document</code> in the repository
	 */
	public synchronized IDocument removeDocument(String document) throws 
			IOException, DocumentNotExistsException {
		try {
			load();
		}
		catch (RepositoryNotExistsException e) {
			throw new IOException("Repository not found.");
		}
		
		validFilename(document);
		
		File file=new File(repoRoot+File.separator+repositoryName+
				File.separator+document+".1");
		if(!file.exists()) {
			throw new DocumentNotExistsException(document);
		}
		
		if(!file.isFile()) {
			throw new IOException("Can't delete a directory.");
		}
		
		/* get highest version */
		long a=getHighestVersion(document);
		
		IDocument retval=getDocument(document, a);
		
		boolean success=true;
		for(long b=1;b<=a;b++) {
			success=(new File(repoRoot+File.separator+repositoryName+
					File.separator+document+"."+Long.toString(b))).delete()
					&& success;
		}
		
		if(!success) {
			throw new IOException("Can't remove document.");
		}
		return retval;
	}

	/**
	 * Sets the users that are allowed to access the repository. The owner of
	 * the repository will always be allowed to access the repository,
	 * independent of the users set using this method.
	 * @param usernames array of the names of the users that shall be allowed
	 * to access the repository
	 */
	public void setAllowedUsers(String[] usernames) {
		this.allowedUsers=usernames;
	}

	/**
	 * Sets the name of the repository
	 * @param name new name of the repository
	 */
	public void setName(String name) {
		this.repositoryName=name;
	}

	/**
	 * Sets the owner of the repository
	 * @param user new owner of the repository
	 */
	public void setOwner(IUser user) {
		this.owner=user;
	}

	/**
	 * Sets the URL of the RMI server
	 * @param url new URL of the RMI server
	 */
	public void setServerUrl(String url) {
		this.serverUrl=url;
	}

	/**
	 * Updates a single document in the repository
	 * @param document document that shall be updated
	 * @throws IOException if the document cannot be updated
	 * @throws FileManException if the version number of the document that shall
	 * be updated is not equal with the version in the repository
	 * @throws DocumentNotExistsException if the document that shall be
	 * updated des not exist in the repository
	 */
	public synchronized void updateDocument(IDocument document) throws
			IOException, FileManException, DocumentNotExistsException {
		load();
		validFilename(document.getName());
		
		if(!(new File(repoRoot+File.separator+repositoryName+File.separator+
				document.getName()+".1")).exists()) {
			throw new DocumentNotExistsException(document.getName());
		}
		
		if(document.getVersion()!=getHighestVersion(document.getName())) {
			throw new FileManException("Invalid version number.");
		}
		
		FileOutputStream output=new FileOutputStream(repoRoot+File.separator+
				repositoryName+File.separator+document.getName()+"."+
				(document.getVersion()+1));
		output.write(document.getContent());
		output.flush();
		output.close();
	}
}
