package at.ac.tuwien.dslab.rmi.server;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.HashMap;

import at.ac.tuwien.dslab.rmi.common.impl.DocumentAlreadyExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.DocumentImpl;
import at.ac.tuwien.dslab.rmi.common.impl.DocumentNotExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.FileManException;
import at.ac.tuwien.dslab.rmi.common.impl.RepositoryExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.RepositoryNamingException;
import at.ac.tuwien.dslab.rmi.common.impl.RepositoryNotExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.UserAccessException;
import at.ac.tuwien.dslab.rmi.common.impl.UserImpl;
import at.ac.tuwien.dslab.rmi.common.interfaces.IDocument;
import at.ac.tuwien.dslab.rmi.common.interfaces.IFileManager;
import at.ac.tuwien.dslab.rmi.common.interfaces.IRepository;
import at.ac.tuwien.dslab.rmi.common.interfaces.IRepositoryListener;
import at.ac.tuwien.dslab.rmi.common.interfaces.IUser;

/**
 * <code>FileManager</code> implements the IFileManager interface.
 * 
 * @author Paul Staroch
 */
public class FileManagerImpl implements IFileManager {
	/**
	 * generated serial version UID
	 */
	private static final long serialVersionUID = 6101885261785405126L;

	/**
	 * List of lists of registered repository listeners for each known
	 * repository.
	 */
	private HashMap<String, ArrayList<IRepositoryListener>> listeners=
		new HashMap<String, ArrayList<IRepositoryListener>>();
	
	/**
	 * List of known repositories
	 */
	private HashMap<String, IRepository> repositories=
		new HashMap<String, IRepository>();
		
	/**
	 * Repository root
	 */
	private String repoRoot;
	
	/**
	 * List of all allowed users
	 */
	private ArrayList<IUser> allowedUsers=new ArrayList<IUser>();
	
	/**
	 * Constructor
	 */
	public FileManagerImpl() {
		repoRoot=Global.getDocumentRoot();

		getAllowedUsers();
		
		/*
		 * check for repositories already existing repositories in the
		 * repository root
		 */ 
		File file=new File(repoRoot);
		for(String filename : file.list()) {
			File entry=new File(repoRoot+File.separator+filename);
			if(entry.isDirectory() && entry.canRead()) {
				File configFile=new File(repoRoot+File.separator+filename+
						File.separator+RepositoryImpl.CONFIG_FILENAME);
				if(configFile.exists() && configFile.canRead()) {
					IRepository repository=null;
					repository = new RepositoryImpl(repoRoot);
					repository.setName(filename);
					try {
						repository.load();
					}
					catch (RepositoryNotExistsException e) {
						/* can't happen */
					}
					catch (IOException e) {
						throw new IllegalArgumentException(e);
					}
				}
			}
		}
	}

	/**
	 * Checks if the specified user is allowed to access the specified
	 * repository 
	 * @param repository repository that the client wants to access
	 * @param user user that wants to access the repository
	 * @throws UserAccessException if the user is not allowed to access the
	 * specified repository
	 */
	private void checkValidUser(IRepository repository, IUser user)
			throws UserAccessException {
		if(!repository.getOwner().equals(user)) {
			boolean found=false;
			
			if(allowedUsers.contains(user)) {
				for(String allowedUser : repository.getAllowedUsers()) {
					if(allowedUser.equals(user.getName())) {
						found=true;
						break;
					}
				}
			}
			
			if(!found) {
				throw new UserAccessException("Permission denied.");
			}
		}
	}
	
	/**
	 * Checks if the repository with the given name is known
	 * @param repoName name of the repository
	 * @throws RepositoryNotExistsException if the repository is not known
	 */
	private void checkRepo(String repoName)
			throws RepositoryNotExistsException {
		if(!repositories.containsKey(repoName)) {
			throw new RepositoryNotExistsException(repoName);
		}
	}
	
	/**
	 * Adds a listener to a repository.
	 * @param repoName name of the repository where to add the repository
	 * listener to.
	 * @param user user that wants to be notified about changes to the
	 * repository <code>repoName</code>.
	 * @param listener listener that shall be added to the repository
	 * <code>repoName</code>.
	 * @throws RepositoryNotExistsException if the repository
	 * <code>repoName</code> does not exist.
	 * @throws UserAccessException if the user <code>user</code> is not allowed
	 * to access the repository <code>repoName</code>.
	 * @throws FileManException if another error occurs.
	 * @throws RemoteException if a communication error occurs.
	 */
	public void addRepositoryListener(String repoName, IUser user,
			IRepositoryListener listener) throws RepositoryNotExistsException,
			UserAccessException, FileManException, RemoteException {
		if(repoName==null || user==null || listener==null) {
			throw new IllegalArgumentException();
		}
		checkRepo(repoName);
		checkValidUser(repositories.get(repoName), user);
		listeners.get(repoName).add(listener);
	}

	/**
	 * Retrieves all documents from the repository.
	 * @param repoName name of the repository.
	 * @param user user that wants to access the files in the repository
	 * <code>repoName</code>.
	 * @return array of documents that are checked out
	 * @throws RepositoryNotExistsException if the repository
	 * <code>repoName</code> does not exist
	 * @throws UserAccessException if the user <code>user</code> is not allowed
	 * to access the repository <code>repoName</code>.
	 * @throws FileManException if another error occurs.
	 * @throws RemoteException if a communication error occurs.
	 */
	public IDocument[] checkout(String repoName, IUser user)
			throws RepositoryNotExistsException, UserAccessException,
			FileManException, RemoteException {
		if(repoName==null || user==null) {
			throw new IllegalArgumentException();
		}
		checkRepo(repoName);
		checkValidUser(repositories.get(repoName), user);
		
		try {
			return repositories.get(repoName).getAllDocuments();
		}
		catch (IOException e) {
			throw new FileManException(e.getMessage(), e);
		}
	}

	/**
	 * Updates a document that is already in the repository.
	 * @param repoName name of the repository
	 * @param user user that wants to update the file
	 * @param document instance of <code>IDocument</code> holding all data
	 * about the document to be updated
	 * @return new version of the document that has been committed
	 * @throws RepositoryNotExistsException if the repository
	 * <code>repoName</code> does not exist
	 * @throws UserAccessException if the user <code>user</code> is not
	 * allowed to access the repository <code>repoName</code>
	 * @throws DocumentNotExistsException if the document <code>document</code>
	 * cannot be found in the repository <code>repoName</code>
	 * @throws FileManException if another error occurs
	 * @throws RemoteException if a connection error occurs
	 */
	public long commitDocument(String repoName, IUser user,
			final IDocument document)
			throws RepositoryNotExistsException, UserAccessException,
			DocumentNotExistsException, FileManException, RemoteException {
		if(repoName==null || user==null || document==null) {
			throw new IllegalArgumentException();
		}
		checkRepo(repoName);
		checkValidUser(repositories.get(repoName), user);
		
		try {
			repositories.get(repoName).updateDocument(document);
		}
		catch (IOException e) {
			throw new FileManException(e.getMessage(), e);
		}
		
		/* notify listeners about the event */
		for(IRepositoryListener listener : listeners.get(repoName)) {
			IDocument notifyDocument=new DocumentImpl(document.getName(),
					document.getVersion()+1, document.getContent());
			listener.documentCommitted(notifyDocument);
		}
		return document.getVersion()+1;
	}

	/**
	 * Retrieves the list of the users that are allowed to access the server.
	 */
	private void getAllowedUsers() {
		BufferedReader reader=new BufferedReader(new InputStreamReader
				(FileManagerImpl.class.getResourceAsStream
						(File.separator+Global.getUserFile())));

		String line;
		try {
			while((line=reader.readLine())!=null) {
				if(!line.equals("")) {
					String username=line.substring(0,line.indexOf(":"));
					String password=line.substring(line.indexOf(":")+1);
					allowedUsers.add(new UserImpl(username, password));
				}
			}
		}
		catch (IOException e) {
			/* may not happen */
		}
	}

	/**
	 * Creates a repository.
	 * @param repoName name of the new repository
	 * @param user owner of the repository
	 * @param allowedUsers names of the users that may access the new repository
	 * @throws RepositoryExistsException if a repository with the name
	 * <code>repoName</code> already exists
	 * @throws RepositoryNamingException if <code>repoName</code> doesn't start
	 * with the name of the owner, followed by an underscore
	 * @throws UserAccessException if the user <code>user</code> is not allowed
	 * to create a repository or if <code>allowedUsers</code> contains the name
	 * of a user that is not allowed to access the server
	 * @throws FileManException if another error occurs
	 * @throws RemoteException if a connection error occurs
	 */
	public void createRepository(String repoName, IUser user,
				String... allowedUsers)
			throws RepositoryExistsException, RepositoryNamingException,
			UserAccessException, FileManException, RemoteException {
		if(repoName==null || user==null || allowedUsers==null) {
			throw new IllegalArgumentException();
		}
		for(String allowedUser : allowedUsers) {
			if(allowedUser==null) {
				throw new IllegalArgumentException();
			}
		}
		if(!this.allowedUsers.contains(user)) {
			throw new UserAccessException("Permission denied.");
		}
		if(!repoName.startsWith(user.getName()+"_")) {
			throw new RepositoryNamingException(repoName);
		}
		IRepository repository=new RepositoryImpl(repoRoot);
		repository.setName(repoName);
		repository.setOwner(user);
		for(String allowedUser : allowedUsers) {
			boolean found=false;
			for(IUser thisAllowedUser : this.allowedUsers) {
				if(thisAllowedUser.getName().equals(allowedUser)) {
					found=true;
					break;
				}
			}
			
			if(!found) {
				throw new UserAccessException("Unknown user "+allowedUser);
			}
		}
		repository.setAllowedUsers(allowedUsers);
		try {
			repository.create();
		}
		catch (IOException e) {
			throw new FileManException(e.getMessage(), e);
		}
		repositories.put(repoName, repository);
		listeners.put(repoName, new ArrayList<IRepositoryListener>());
	}

	/**
	 * Imports a new document into a repository.
	 * @param repoName the name of the repository where to add the document
	 * <code>document</code>
	 * @param user the user that wants to import the new document
	 * @param document the document that is to be added to the repository
	 * <code>repoName</code>
	 * @throws RepositoryNotExistsException if the repository
	 * <code>repoName</code> does not exist
	 * @throws UserAccessException if the user <code>user</code> is not
	 * allowed to import a new file into the repository <code>repoName</code>
	 * @throws DocumentAlreadyExistsException if the document that is to be
	 * imported already exists in the repository <code>repoName</code>
	 * @throws FileManException if another error occurs
	 * @throws RemoteException if a communication error occurs
	 */
	public void importDocument(String repoName, IUser user, IDocument document)
			throws RepositoryNotExistsException, UserAccessException,
			DocumentAlreadyExistsException, FileManException, RemoteException {
		if(repoName==null || user==null || document==null) {
			throw new IllegalArgumentException();
		}
		checkRepo(repoName);
		checkValidUser(repositories.get(repoName), user);
		
		try {
			repositories.get(repoName).createDocument(document);
		}
		catch (IOException e) {
			throw new FileManException(e.getMessage(), e);
		}
		
		/* notify the registered listeners about the event */
		for(IRepositoryListener listener : listeners.get(repoName)) {
			listener.documentImported(document);
		}
	}

	/**
	 * Permanently removes a document from a repository
	 * @param repoName name of the repository where to remove the document
	 * with the name <code>filename</code> from
	 * @param user user that wants to remove the document with the name
	 * <code>filename</code>
	 * @param filename name of the document that is to be removed from the
	 * repository <code>repoName</code>
	 * @throws RepositoryNotExistsException if the repository
	 * <code>repoName</code> does not exist
	 * @throws UserAccessException if the user <code>user</code> is not
	 * allowed to remove the document from the repository  
	 * @throws FileManException if another error occurs
	 * @throws RemoteException if a communication error occurs
	 */
	public void removeDocument(String repoName, IUser user, String filename)
			throws RepositoryNotExistsException, UserAccessException,
			FileManException, RemoteException {
		if(repoName==null || user==null || filename==null) {
			throw new IllegalArgumentException();
		}

		checkRepo(repoName);
		checkValidUser(repositories.get(repoName), user);
		
		IDocument document;
		try {
			document = repositories.get(repoName).getDocument(filename, -1);
			repositories.get(repoName).removeDocument(filename);
		}
		catch (IOException e) {
			return;
		}
		
		/* notify the registered listeners about the event */
		for(IRepositoryListener listener : listeners.get(repoName)) {
			listener.documentRemoved(document);
		}
	}

	/**
	 * Permanently removes a repository and all of its content
	 * @param repoName name of the repository that is to be deleted
	 * @param user user that wants to delete the repository
	 * @throws RepositoryNotExistsException if the repository
	 * <code>repoName</code> does not exist
	 * @throws UserAccessException if the user <code>user</code> is not
	 * allowed to remove the repository <code>repoName</code>
	 * @throws FileManException if another error occurs
	 * @throws RemoteException if a communication error occurs
	 */
	public void removeRepository(String repoName, IUser user)
			throws RepositoryNotExistsException, UserAccessException,
			FileManException, RemoteException {
		if(repoName==null || user==null) {
			throw new IllegalArgumentException();
		}
		checkRepo(repoName);
		try {
			repositories.get(repoName).delete(user);
		}
		catch (IOException e) {
			throw new FileManException(e.getMessage(), e);
		}
		repositories.remove(repoName);
		listeners.remove(repoName);
	}

	/**
	 * Removes a repository listener from a repository
	 * @param repoName name of the repository where to remove the listener from
	 * @param listener instance of <code>IRepositoryListener</code> that
	 * previously has been added to the repository <code>repoName</code>
	 * using <code>addRepositoryListener</code> and shall be removed now
	 * @throws RepositoryNotExistsException if the repository
	 * <code>repoName<code> does not exist
	 * @throws FileManException if another error occurs
	 * @throws RemoteException if a communication error occurs
	 */
	public void removeRespositoryListener(String repoName,
			IRepositoryListener listener) throws RepositoryNotExistsException, 
			FileManException, RemoteException {
		if(repoName==null || listener==null) {
			throw new IllegalArgumentException();
		}
		checkRepo(repoName);
		listeners.get(repoName).remove(listener);
	}

	/**
	 * Retrieves all documents from a repository.
	 * @param repoName name of the repository where to get all files from 
	 * @param user user that wants to get all the files from the repository
	 * <code>repoName</code>
	 * @return array containing all documents in the repository
	 * <code>repoName</code>
	 * @throws RepositoryNotExistsException if the repository
	 * <code>repoName</code> does not exist
	 * @throws UserAccessException if the user <code>user</code> is not
	 * allowed to access the files located in the repository
	 * <code>repoName</code>
	 * @throws FileManException if another error occurs
	 * @throws RemoteException if a communication error occurs
	 */
	public IDocument[] updateAllDocuments(String repoName, IUser user)
			throws RepositoryNotExistsException, UserAccessException,
			FileManException, RemoteException {
		return checkout(repoName, user);
	}

	/**
	 * Retrieves a single document from a repository.
	 * @param repoName name of the repository where to get the document named
	 * <code>document</code> from
	 * @param user user that wants to retrieve the file from the repository
	 * @param document name of the document to be retrieved from the
	 * repository <code>repoName</code>
	 * @param version version of the document to be retrieved from the
	 * repository <code>repoName</code>
	 * @return document that has been fetched from the repository
	 * @throws RepositoryNotExistsException if the repository
	 * <code>repoName</code> does not exist
	 * @throws UserAccessException if the user <code>user</code> is not
	 * allowed to fetch a file from the repository <code>repoName</code>
	 * @throws FileManException if another error occurs
	 * @throws RemoteException if a communication error occurs
	 */
	public IDocument updateDocument(String repoName, IUser user,
			String document, long version) throws RepositoryNotExistsException,
			UserAccessException, FileManException, RemoteException {
		if(repoName==null || user==null || document==null) {
			throw new IllegalArgumentException();
		}
		checkRepo(repoName);
		checkValidUser(repositories.get(repoName), user);
		try {
			return repositories.get(repoName).getDocument(document, version);
		}
		catch (IOException e) {
			throw new FileManException(e.getMessage(), e);
		}
	}

}
