/*
 * Distributed Systems
 * Winter Term 2006/07
 * Lab 3
 * Paul Staroch, 0425426
 */

package at.ac.tuwien.dslab.rmi.client.command;

import at.ac.tuwien.dslab.rmi.common.impl.FileManException;
import at.ac.tuwien.dslab.rmi.common.impl.InvalidCommandException;
import at.ac.tuwien.dslab.rmi.common.interfaces.IFileManager;

import java.rmi.Naming;
import java.rmi.NotBoundException;
import java.net.MalformedURLException;
import java.rmi.RemoteException;

/**
 * Command represents an abstract class that will be extended by another
 * class that processes a certain command.
 */
public abstract class Command {
	/**
	 * minimum number of arguments for the method <code>execute</code>
	 * will be set in the constructor of any subclass
	 */
	protected int minArgumentNumber;
	
	/**
	 * maximum number of arguments for the method <code>execute</code>
	 * will be set in the constructor of any subclass
	 */
	protected int maxArgumentNumber;

	/**
	 * performs the command, but doesn't check if the number of
	 * arguments is correct (like <code>execute</code> does)
	 *
	 * @param args arguments to the command
	 * @return message for the user (about success or failure)
	 * @throws FileManException if an error occurs
	 */
	protected abstract String executeNoCheckNumber(String... args)
		throws FileManException;

	/**
	 * performs the command
	 *
	 * @param args arguments to the command
	 * @throws InvalidCommandException if the arguments are incorrect
	 * @throws FileManException if an error occurs
	 */
	public String execute(String... args)
	       		throws FileManException {
		if(!checkArgumentNumber(args)) {
			throw new InvalidCommandException
				("Invalid number of arguments.");
		}
		
		return executeNoCheckNumber(args);
	}

	/**
	 * initializes the file manager to be used for communication with the
	 * server
	 *
	 * @param host RMI URL to the server
	 * @return instance of IFileManager
	 * @throws FileManException if an error occurs
	 */
	protected final IFileManager getFileManager(String host)
			throws FileManException {
		IFileManager fileManager;
		try {
			return (IFileManager)Naming.lookup(host);
		}
		catch (NotBoundException nbe) {
			throw new FileManException
				("Name "+host+" is currently not bound.",
				 nbe);
		}
		catch (MalformedURLException mue) {
			throw new FileManException
				("Malformed URL.", mue);
		}
		catch (RemoteException re) {
			throw new FileManException
				("Cannot contact RMI registry.", re);
		}
	}

	/**
	 * checks if the given arguments have a correct number (between
	 * <code>minArgumentNumber</code> and <code>maxArgumentNumber</code>,
	 * as set by the subclass' constructor.
	 *
	 * @param args
	 * @return <code>true</code> if the number of arguments is correct,
	 * 	<code>false</code> otherwise
	 */ 
	private final boolean checkArgumentNumber(String... args) {
		return (args.length>=minArgumentNumber) &&
		       	((maxArgumentNumber==-1) ||
			 (args.length<=maxArgumentNumber));
	}
}

