/*
 * Distributed Systems
 * Winter Term 2006/07
 * Lab 3
 * Paul Staroch, 0425426
 */

package at.ac.tuwien.dslab.rmi.client;

import at.ac.tuwien.dslab.rmi.common.impl.DocumentNotExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.DocumentAlreadyExistsException;
import at.ac.tuwien.dslab.rmi.common.impl.FileManException;

import java.util.HashMap;
import java.io.File;
import java.io.FileWriter;
import java.io.FileReader;
import java.io.BufferedReader;
import java.io.IOException;
import java.util.Set;

/**
 * FileManClientConfig encapsulates the configuration stored in a file in
 * the local working directory. This includes:
 * <ul>
 * <li>The username used for accessing the repository.
 * <li>The password for this user.
 * <li>The host (in other words, the URL to the RMI server) of the repository.
 * <li>The name of the repository.
 * <li>The names and versions of all files that are stored in the repositor
 *   and in the local working copy of the repository.
 * </ul>
 *
 * @author	Paul Staroch
 * @version	1.0
 */
public class FileManClientConfig {
	/**
	 * working directory of the repository
	 */
	private String workingDir;

	/**
	 * name of the file where to store the configuration
	 */
	private final String configFileName=".repository";

	/**
	 * message to be printed when not being able to write the
	 * configuration file
	 */
	private final String cantReadConfig="Can't read config file.";

	/**
	 * username to be used to access the repository
	 */
	private String username="";

	/**
	 * password to be used to access the repository
	 */
	private String password="";

	/**
	 * RMI server of the repository
	 */
	private String host="";

	/**
	 * name of the repository on the server
	 */
	private String repository="";

	/**
	 * HashMap containing the names and versions of all files in the local
	 * working copy of the repository that belong to the repository
	 */
	private HashMap<String, Long> files=new HashMap<String, Long>();

	/**
	 * Class constructor specifying the local working copy of the
	 * repository
	 * 
	 * @param workingDir path to the local working copy of the repository
	 * @throws FileManConfigException when an error occurs while reading
	 * 	the configuration file in <code>workingDir</code>
	 */
	public FileManClientConfig(String workingDir)
			throws FileManConfigException {
		this(workingDir, true);
	}

	/**
	 * Class constructor specifying the local working copy of the
	 * repository
	 *
	 * @param workingDir path to the local working copy of the repository
	 * @param readConfig whether to read an already existing configuration
	 * 	file in <code>workingDir</code> or not
	 * @throws FileManConfigException when an error occurs while reading
	 * 	the configuration file in <code>workingDir</code> (only if
	 * 	<code>readConfig</code> is set to <code>true</code>)
	 */
	public FileManClientConfig(String workingDir, boolean readConfig) 
			throws FileManConfigException {
		this.workingDir=workingDir;

		if(readConfig) {
			readConfig();
		}
	}

	/**
	 * finds out if a configuration file already exists in the working
	 * directory
	 * 
	 * @return <code>true</code> ifa configuration file exists,
	 * 	<code>false</code> otherwise
	 */
	public boolean configFileExists() {
		File file=new File(workingDir+File.separator+configFileName);
		return file.exists();
	}

	/**
	 * returns the username for accessing the repository
	 *
	 * @return username
	 */
	public String getUsername() {
		return username;
	}

	/**
	 * returns the password for accessing the repository
	 *
	 * @return password
	 */
	public String getPassword() {
		return password;
	}

	/**
	 * returns the host of the RMI server that holds the repository
	 *
	 * @return host
	 */
	public String getHost() {
		return host;
	}

	/**
	 * returns the name of the repository on the server
	 *
	 * @return repository name
	 */
	public String getRepository() {
		return repository;
	}

	/**
	 * changes the username for accessing the repository
	 *
	 * @param username username that shall be used
	 */
	public void setUsername(String username) {
		this.username=username;
	}

	/**
	 * changes the password for accessing the repository
	 *
	 * @param password password that shall be used
	 */
	public void setPassword(String password) {
		this.password=password;
	}

	/**
	 * changes the host of the RMI server that holds the repository
	 *
	 * @param host host that shall be used
	 */
	public void setHost(String host) {
		this.host=host;
	}

	/**
	 * changes the name of the repository on the server that is being
	 * used for accessing the repository
	 *
	 * @param repository repository name that shall be used
	 */
	public void setRepository(String repository) {
		this.repository=repository;
	}

	/**
	 * adds a file to the file list
	 *
	 * @param filename name of the file to be added
	 * @param version this file's version
	 * @throws DocumentAlreadyExistsException if the file list already
	 * 	contains a file named <code>filename</code>
	 */
	public void addFile(String filename, long version)
			throws DocumentAlreadyExistsException {
		if(files.containsKey(filename)) {
			throw new DocumentAlreadyExistsException(filename);
		}

		files.put(filename, version);
	}

	/**
	 * removes a file from the file list
	 *
	 * @param filename name of the file to be removed
	 * @throws DocumentNotExistsException if the list does not contain a
	 * 	file named <code>filename</code>
	 */
	public void deleteFile(String filename)
			throws DocumentNotExistsException {
		if(!files.containsKey(filename)) {
			throw new DocumentNotExistsException(filename);
		}

		files.remove(filename);
	}

	/**
	 * updates the version of a file in the file list
	 *
	 * @param filename the name of the file that is to be updated
	 * @param version new version of this file
	 * @throws FileManException if the file list does not contain a
	 * 	file named <code>filename</code>
	 */
	public void updateFile(String filename, long version)
			throws FileManException {
		deleteFile(filename);
		addFile(filename, version);
	}

	/**
	 * retrieves a <code>Set</code> containing the names of all files in
	 * the local working directory that belong to the repository
	 *
	 * @return <code>Set</code> containing the filenames
	 */
	public Set<String> getFiles() {
		return files.keySet();
	}

	/**
	 * deletes the config file in the local working directory
	 *
	 * @return <code>true</code> if the file has been deleted
	 * 	successfully, <code>false</code> otherwise
	 */
	public boolean deleteConfig() {
		File file=new File(workingDir+File.separator+configFileName);
		
		return file.delete();
	}
	
	/**
	 * checks if the file list contains a file that is specified by its
	 * name
	 *
	 * @param filename name of the file to check
	 * @return <code>true</code> if the file list contains a file named
	 * 	<code>filename</code>, <code>false</code> otherwise
	 */
	public boolean containsFile(String filename) {
		return files.containsKey(filename);
	}

	/**
	 * retrieves the version of a file in the file list
	 *
	 * @param filename name of the file
	 * @return current version of the file named <code>filename</code>
	 * @throws DocumentNotExistsException if the file list does not contain
	 * 	a file named <code>filename</code>
	 */
	public long getVersion(String filename)
			throws DocumentNotExistsException {
		if(!files.containsKey(filename)) {
			throw new DocumentNotExistsException(filename);
		}

		return files.get(filename);
	}

	/**
	 * reads a single line from a <code>BufferedReader</code>
	 *
	 * @param inputStream <code>BufferedReader</code> where to read from
	 * @param whatToRead a short string that will be included in the
	 * 	exception's error message in case an error occurs
	 * @return <code>String</code> that has been read from
	 * 	<code>inputStream</code>
	 * @throws FileManConfigException if an error occurs when reading
	 * 	from <code>inputStream</code>
	 */
	private String readInput(BufferedReader inputStream, String whatToRead)
			throws FileManConfigException {
		String input;

		try {
			input=inputStream.readLine();
		}
		catch(IOException ioe) {
			try {
				inputStream.close();
			}
			catch (IOException ie) {
				/* do nothing, just inform about the error */
			}
			throw new FileManConfigException(cantReadConfig, ioe);
		}
		if(input==null) {
			try {
				inputStream.close();
			}
			catch (IOException ioe) {
				/* do nothing, just inform about the error */
			}
			throw new FileManConfigException("Can't read "+
					whatToRead+" from config file.");
		}

		return input;
	}
	
	/**
	 * parses the config file in the current working directory
	 *
	 * @throws FileManConfigException if an error occurs
	 */
	public void readConfig() throws FileManConfigException {
		BufferedReader inputStream;
		
		/* open stream */
		try {
			inputStream=new BufferedReader(new FileReader
					(workingDir+File.separator+
					 configFileName));
		}
		catch(IOException ioe) {
			throw new FileManConfigException(cantReadConfig, ioe);
		}
		
		if(inputStream==null) {
			throw new FileManConfigException(cantReadConfig);
		}
		String input;

		/*
		 * read information for accessing the repository on the server
		 */
		this.username=readInput(inputStream, "username");
		this.password=readInput(inputStream, "password");
		this.host=readInput(inputStream, "host");
		this.repository=readInput(inputStream, "repository");
	
		/* read files in repository and their versions */
		try {
			while((input=inputStream.readLine())!=null) {
				/* ignore empty lines */
				if(!input.equals("")) {
					String filename=input.substring
						(0,input.lastIndexOf(" "));
					String versionString=input.substring
						(input.lastIndexOf(" ")+1);

					long version;
					try {
						version=Long.parseLong
							(versionString);
					}
					catch (NumberFormatException nfe) {
						inputStream.close();
						throw new
							FileManConfigException
							("Version number is "+
							 "not an integer.",
							 nfe);
					}
	
					files.put(filename, version);
				}
			}
		}
		catch (IOException ioe) {
			try {
				inputStream.close();
			}
			catch (IOException ie) {
				/* do nothing */
			}

			throw new FileManConfigException
				("Can't read config file.", ioe);
		}

		/* close stream */
		try {
			inputStream.close();
		}
		catch (IOException ioe) {
			throw new FileManConfigException
				("Can't read config file.", ioe);
		}

		/* we're done */
	}

	/**
	 * stores the configuration in the configuration file in the current
	 * working dirctory
	 *
	 * @throws IOException if an error occurs
	 */
	public void writeConfig() throws IOException {
		FileWriter writer=new FileWriter
			(workingDir+File.separator+configFileName);
		
		writer.write(username+"\n");
		writer.flush();

		writer.write(password+"\n");
		writer.flush();

		writer.write(host+"\n");
		writer.flush();
		
		writer.write(repository+"\n");
		writer.flush();
		
		for (String key : files.keySet()) {
			writer.write(key+" "+files.get(key)+"\n");
			writer.flush();
		}

		writer.close();
	}
}

