/*
 * Distributed Systems
 * Winter Term 2006/07
 * Lab 3
 * Paul Staroch, 0425426
 */

package at.ac.tuwien.dslab.rmi.client.command;

import at.ac.tuwien.dslab.rmi.client.command.Command;
import at.ac.tuwien.dslab.rmi.common.interfaces.IFileManager;
import at.ac.tuwien.dslab.rmi.common.interfaces.IDocument;
import at.ac.tuwien.dslab.rmi.common.impl.DocumentImpl;
import at.ac.tuwien.dslab.rmi.common.impl.UserImpl;
import at.ac.tuwien.dslab.rmi.client.FileManClientConfig;
import at.ac.tuwien.dslab.rmi.common.impl.FileManException;
import at.ac.tuwien.dslab.rmi.common.impl.InvalidCommandException;
import at.ac.tuwien.dslab.rmi.client.FileManConfigException;

import java.io.FileOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Set;
import java.util.TreeSet;

import java.rmi.RemoteException;

/**
 * UpdateDocumentCommand represents the command "update"
 */
public class UpdateDocumentCommand extends Command {
	/**
	 * Constructor - sets the minimum and maximum argument count
	 */
	public UpdateDocumentCommand() {
		super();
		minArgumentNumber=0;
		maxArgumentNumber=2;
	}

	/**
	 * performs the command
	 *
	 * @param args arguments to the command
	 * @return message for the user (about success or failure)
	 * @throws InvalidCommandException if the command fails due to wrong
	 *	arguments
	 * @throws FileManException if an error occurs
	 */
	public String executeNoCheckNumber(String... args)
			throws FileManException {
		/* read configuration */
		FileManClientConfig config;
		try {
			config=new FileManClientConfig(".");
		}
		catch(FileManConfigException fmce) {
			throw new InvalidCommandException("Can't read "+
					"repository config file.",fmce);
		}

		if(!config.configFileExists()) {
			throw new InvalidCommandException
				("No repository config file found.");
		}
		
		IFileManager fileMan;
		try {
			fileMan=getFileManager(config.getHost());
		}
		catch (FileManException fme) {
			return fme.getMessage();
		}

		UserImpl user=new UserImpl();
		user.setName(config.getUsername());
		user.setPasswordHashed(config.getPassword());

		if(args.length==0) {
			/* update all documents */
			IDocument[] documents;
			try {
				documents=fileMan.updateAllDocuments
					(config.getRepository(),user);
			}
			catch (RemoteException re) {
				return "Communication error";
			}

			/*
			 * currentFilenames stores the names of all files
			 * currently being located in the repository on the
			 * server
			 */
			ArrayList<String> currentFilenames=
				new ArrayList<String>();
			
			for (IDocument document : documents) {
				currentFilenames.add(document.getName());

				String filename=document.getName();

				/* write file to working directory */
				try {
					FileOutputStream output=
						new FileOutputStream(filename);
					output.write(document.getContent());
					output.flush();
					output.close();
				}
				catch (Exception e) {
					return "Error while writing file "+
						filename+".";
				}

				/* add/update file in local configuration */
				if(!config.getFiles().contains
						(document.getName())) {
					config.addFile(document.getName(),
							document.getVersion());
				}
				else {
					config.updateFile(document.getName(),
							document.getVersion());
				}
			}

			/*
			 * avoid ConcurrentModificationException in the
			 * for-each-loop below
			 *
			 * don't know exactly what this is necessary for
			 */
			Set<String> files=new TreeSet<String>();
			for (String filename : config.getFiles()) {
				files.add(new String(filename));
			}

			/*
			 * delete all files from the working directory that
			 * no longer exist in the repository
			 */
			for (String filename : files) {
				if(!currentFilenames.contains(filename)) {
					File file=new File(filename);
					file.delete();

					config.deleteFile(filename);
				}
			}
		}
		else {
			/* update single document */
			if(args[0].indexOf(File.separator)>-1) {
				throw new InvalidCommandException
					("The file you want to update is not "+
					 "located in the working directory.");
			}
			
			/* certain version? */
			long version=-1;
			if(args.length==2) {
				try {
					version=Long.parseLong(args[1]);
				}
				catch (NumberFormatException nfe) {
					throw new InvalidCommandException
						("Invalid version number");
				}
			}

			File file=new File(args[0]);
/*			if(!file.exists()) {
				throw new InvalidCommandException
					("The file you want to update does "+
					 "not exist.");
			}
			if(!file.isFile()) {
				throw new InvalidCommandException
					("The file you want to update is "+
					 "not a normal file.");
			}*/
			
			/* fetch file */
			IDocument document;
			try {
				document=fileMan.updateDocument
					(config.getRepository(),user,
					 args[0],version);
			}
			catch (RemoteException re) {
				throw new FileManException
					("Communication error", re);
			}
			
			if(document==null) {
				/* file has been deleted on the server */
				file.delete();

				config.deleteFile(args[0]);
			}
			else {
				/* update file */
				String filename=document.getName();

				try {
					FileOutputStream output=
						new FileOutputStream(filename);
					output.write(document.getContent());
					output.flush();
					output.close();
				}
				catch (IOException ioe) {
					throw new FileManException("Error "+
							"while writing file.",
							ioe);
				}

				/*
				 * add/update file in the local
				 * configuration
				 */
				if(!config.getFiles().contains
						(document.getName())) {
					config.addFile(document.getName(),
							document.getVersion());
				}
				else {
					config.updateFile(document.getName(),
							document.getVersion());
				}
			}
		}

		/* write local configuration */
		try {
			config.writeConfig();
		}
		catch (IOException ioe) {
			throw new FileManException("Error while updating "+
					"repository configuration file.");
		}
		return "File"+(args.length==0?"s":"")+" successfully updated.";
	}
}

