/* Uebersetzerbau, SS 2006
 * Paul Staroch, 0425426
 * Gesamtbeispiel
 *
 * this file contains procedure needed by code.bfe for code generation
 */

#include <stdio.h>
#include <assert.h>
#include <stdlib.h>

#include "tree.h"
#include "code_gen.h"

int regstack_push(int reg) {
	struct intstack *new=(struct intstack *)malloc(sizeof(struct intstack));
	new->next=reg_stack;
	new->value=reg;
	reg_stack=new;
	
#ifdef DEBUG
	printf("reg_push: %i\n",reg);
#endif

	return reg;
}

/* regstack_pop: takes the number of a register from the register stack */
int regstack_pop(void) {
	int reg;
	struct intstack *pop_element=reg_stack;

	if(reg_stack==(struct intstack *)NULL) {
		fprintf(stderr,"Internal error 1\n");
		exit(5);
	}

	reg_stack=pop_element->next;
	reg=pop_element->value;
	(void)free(pop_element);

	if(reg<0) {
		reg=-reg;
		fprintf(output,"\tldq $%i, 0($sp)\n",reg);
		fprintf(output,"\taddq $sp, 8, $sp\n");
	}

#ifdef DEBUG
	printf("reg_pop: %i\n",reg);
#endif

	return reg;
}

/* copy_assignments: copies a list of register assignments */
struct register_assignment *copy_assignments(struct register_assignment *to_be_copied) {
	struct register_assignment *element=to_be_copied;
	struct register_assignment *new_list=(struct register_assignment *)NULL;
	struct register_assignment *new_previous;
	struct register_assignment *new_current=(struct register_assignment *)NULL;
	
	while(element!=(struct register_assignment *)NULL) {
		new_previous=new_current;
		new_current=(struct register_assignment *)malloc(sizeof(struct register_assignment));
		new_current->name=(char *)malloc(strlen(element->name)+1);
		(void)strcpy(new_current->name,element->name);
		new_current->reg=element->reg;
		new_current->next=(struct register_assignment *)NULL;
		if(new_list==(struct register_assignment *)NULL) {
			new_list=new_current;
		}
		else {
			new_previous->next=new_current;
		}

		element=element->next;
	}

	return new_list;
}

/* free_assignments: destroys a list of register assignments */
void free_assignments(struct register_assignment *to_be_deleted) {
	struct register_assignment *element=to_be_deleted;
	struct register_assignment *temp;
	
	while(element!=(struct register_assignment *)NULL) {
		/* segfault ?! */
		/* (void)free(element->name); */
		temp=element;
		element=element->next;
		(void)free(temp);
	}
}

/* reg_save: save register usage at beginning of loop or conditional statements */
void reg_save(void) {
	struct reg_data *new=(struct reg_data *)malloc(sizeof(struct reg_data));
	new->next=reg_data_stack;
	new->assignments=copy_assignments(register_assignments);
	(void)memcpy(&new->reg_used[0],&reg_used[0],32*sizeof(int));
	reg_data_stack=new;
}

/* reg_restore: restore register usage at end of loop or conditional statements
 * previously saved by reg_save
 */
void reg_restore(void) {
	struct reg_data *element=reg_data_stack;
	struct register_assignment *old_assignments;
	
	if(reg_data_stack==(struct reg_data *)NULL) {
		return;
	}

	(void)memcpy(&reg_used[0],&reg_data_stack->reg_used[0],32*sizeof(int));
	reg_data_stack=reg_data_stack->next;

	old_assignments=register_assignments;
	register_assignments=element->assignments;
	free_assignments(old_assignments); 

	/* TODO: program gets stuck when calling free() */
	/* (void)free(element); */
}

/* setvalues: get values at assignment operator (->) from given node if set;
 * necessary for consumer_reg to know how many values are flowing from left to
 * right at an assignment operator
 */
void setvalues(treenode *node) {
	if(node==(treenode *)NULL) {
		fprintf(stderr,"Internal error 2\n");
		exit(5);
	}
	if(node->values>0) {
		valuecount=node->values;
	}
}

/* consumer_reg: get correct register number for consumer; necessary due to
 * use of a stack for register usage and the possibility of more than a single
 * value flowing from left to right at an assignment operator (->). Without that
 * routine the code
 * 1,2 -> a,b;
 * will set a to 2 and b to 1.
 */
int consumer_reg(void) {
	int reg;
	int a;

	struct intstack *element=reg_stack;
	struct intstack *previous=(struct intstack *)NULL;

	struct intstack *temp;
	int found;
	
	if(element==(struct intstack *)NULL) {
		fprintf(stderr,"Internal error 3\n");
		exit(5);
	}

	for(a=1;a<valuecount;a++) {
		previous=element;
		element=element->next;
		if(element==(struct intstack *)NULL) {
			fprintf(stderr,"Internal error 4\n");
			exit(5);
		}
	}
	reg=element->value;
	
	/* only remove from reg_stack when not a loop register */
	temp=loop_input_stack;
	found=0;
	while(temp!=(struct intstack *)NULL) {
		if(temp->value==abs(reg)) {
			found=1;
			break;
		}
		temp=temp->next;
	}

	if(!found) {
		if(reg<0) {
			reg=-reg;
			fprintf(output,"\taddq $sp, 8, $sp\n");
		}
			
		if(previous==(struct intstack *)NULL) {
			reg_stack=element->next;
		}
		else {
			previous->next=element->next;
		}
		(void)free(element);
	}

	/* consume one value */
	valuecount--;

	return reg;
}

/* do_call: calls _do_call and stores the return value in a register
 */
void do_call(int values, char *name) {
	_do_call(values,name,1);
}

/* do_zero_call: calls _do_call and doesn't store the return value
 */
void do_zero_call(int values, char *name) {
	_do_call(values,name,0);
}

/* _do_call: generate all code necessary for executing a function call
 * number of parameters of function being called is given by values,
 * name is the name of the called function.
 */
void _do_call(int values, char *name, int push) {
	int a;
	struct intstack *element;
	struct intstack *tempstack=(struct intstack *)NULL;
	struct intstack *new;

	for(a=0;a<values;a++) {
		move(consumer_reg(),16+a);
	}
	
	/* save register values on stack */
	element=reg_stack;
	while(element!=(struct intstack *)NULL) {
		new=(struct intstack *)malloc(sizeof(struct intstack *));
		new->value=element->value;
		new->next=tempstack;
		tempstack=new;
		if(element->value>0) {
			fprintf(output,"\tlda $sp, -8($sp)\n");
			fprintf(output,"\tstq $%i, 0($sp)\n",element->value);
		}
		element=element->next;
	}
	fprintf(output,"\tlda $sp, -8($sp)\n");
	fprintf(output,"\tstq $26, 0($sp)\n");
	fprintf(output,"\tjsr $26, %s\n",name);
	fprintf(output,"\tldq $26, 0($sp)\n");
	fprintf(output,"\taddq $sp, 8, $sp\n");
	if(push) {
		move(0,PUSH);
	}

	/* restore register values from stack */
	element=tempstack;
	while(element!=(struct intstack *)NULL) {
		if(element->value>0) {
			fprintf(output,"\tldq $%i, 0($sp)\n",element->value);
			fprintf(output,"\taddq $sp, 8, $sp\n");
		}
		element=element->next;
	}
}

/* int_push: push value onto a stack of integers */
struct intstack *int_push(struct intstack **stack, int value) {
	struct intstack *new=(struct intstack *)malloc(sizeof(struct intstack *));
	
	if(stack==(struct intstack **)NULL) {
		fprintf(stderr,"Internal error 5\n");
		exit(5);
	}

	new->next=*stack;
	new->value=value;
	*stack=new;

	return *stack;
}

/* int_pop: pop value from a stack of integers */
int int_pop(struct intstack **stack) {
	struct intstack *old=*stack;
	int value;

	if(stack==(struct intstack **)NULL || *stack==(struct intstack *)NULL) {
		fprintf(stderr,"Internal error 6\n");
		exit(5);
	}
	
	value=(*stack)->value;
	*stack=(*stack)->next;
	free(old);

	return value;
}

/* enter_if: set actions necessary when entering an if block */
int enter_if(void) {
	maxif++;
	ifstack=int_push(&ifstack,ifcounter);
	ifcounter=maxif;

	return ifcounter;
}

/* exit_if: set actions necessary when exiting an if block */
int exit_if(void) {
	ifcounter=int_pop(&ifstack);

	return ifcounter;
}

/* enter_not: set actions necessary when entering a negation */
int enter_not(void) {
	maxneg++;
	negstack=int_push(&negstack,negcounter);
	negcounter=maxneg;

	return negcounter;
}

/* exit_not: set actions necessary when exiting a negation */
int exit_not(void) {
	fprintf(output,"\nENDNOT_%i:\n",negcounter);
	
	negcounter=int_pop(&negstack);

	fprintf(output,"\tmov 1,$%i\n",PUSH);
	fprintf(output,"\tsubq $%i, $%i, $%i\n",POP,POP,PUSH);
	
	return negcounter;
}

/* enter_loop: set actions necessary when entering a loop */
int enter_loop(treenode *node) {
	maxloop++;
	loopstack=int_push(&loopstack,loopcounter);
	loopcounter=maxloop;

	reg_save();

	set_loop_registers(node);
	fprintf(output,"\nSTARTLOOP_%i:\n",loopcounter);

	return loopcounter;
}

/* exit_loop: set actions necessary when exiting a loop */
int exit_loop(treenode *node) {
	setvalues(node);
	loop_registers(node);
	fprintf(output,"\tbr STARTLOOP_%i\n\nENDLOOP_%i:\n",loopcounter,loopcounter);
	
	loopcounter=int_pop(&loopstack);

	reg_restore(); 
	
	push_loop_registers();
	
	return loopcounter;
}

/* set_if_registers: fix registers for output of an if block and reserve these
 * registers
 */
void set_if_registers(treenode *node) {
	int a;
	struct intstack *new;

	if(node==(treenode *)NULL) {
		fprintf(stderr,"Internal error 7\n");
		exit(5);
	}

	for(a=0;a<node->values;a++) {
		new=(struct intstack *)malloc(sizeof(struct intstack));
		new->next=if_stack;
		new->value=newreg();
		if_stack=new;
	}
}

/* push_if_registers: put the registers that hold the output values of an if
 * block on top of the registers that hold the values to be consumed by the
 * next statement
 */
void push_if_registers(void) {
	int reg;
	int a,b;

	struct intstack *element;
	struct intstack *previous;

	if(if_stack==(struct intstack *)NULL) {
		fprintf(stderr,"Internal error 8\n");
		exit(5);
	}

	for(a=valuecount;a>0;a--) {
		element=if_stack;
		previous=(struct intstack *)NULL;

		for(b=1;b<a;b++) {
			previous=element;
			element=element->next;
			if(element==(struct intstack *)NULL) {
				fprintf(stderr,"Internal error 9\n");
				exit(5);
			}
		}

		reg=element->value;
		if(previous==(struct intstack *)NULL) {
			if_stack=element->next;
		}
		else {
			previous->next=element->next;
		}
		(void)free(element);

		regstack_push(reg);
	}
}

/* if_output: put the values that are flowing from left to right at an
 * assignment operator at the end of an if- or else-branch into the registers
 * previously reserved for the output of the if block by set_if_registers()
 */
void if_output(void) {
	int reg;
	int a,b;

	struct intstack *element;
	struct intstack *previous;

	int valuecount_save=valuecount;
	for(a=valuecount_save;a>0;a--) {
		element=if_stack;

		for(b=1;b<a;b++) {
			element=element->next;
			if(element==(struct intstack *)NULL) {
				fprintf(stderr,"Internal error 10\n");
				exit(5);
			}
		}

		move(consumer_reg(),element->value);
	}
}

/* set_loop_registers: fix registers for output of a loop and reserve these
 * registers
 */
void set_loop_registers(treenode *node) {
	int a;
	struct intstack *new;
	int reg;
	int valuecount_safe;

	struct intstack *push_regs=(struct intstack *)NULL;
	struct intstack *new_push_reg=(struct intstack *)NULL;
	struct intstack *element;

	if(node==(treenode *)NULL) {
		fprintf(stderr,"Internal error 11\n");
		exit(5);
	}
	
	/* reserve registers for values flow from the end of a loop
	 * to the first consumer(s) after the loop */
	for(a=0;a<node->values;a++) {
		new=(struct intstack *)malloc(sizeof(struct intstack));
		new->next=loop_stack;
		new->value=newreg();
		loop_stack=new;
	}

	loop_input_values=int_push(&loop_input_values,valuecount);
	valuecount_safe=valuecount;

	/* move values that flow into the beginning of the loop into the
	 * registers that hold the values that flow from the end of the loop
	 * back to its beginning; these registers are reserved at this point
	 */
	for(a=0;a<valuecount_safe;a++) {
		new=(struct intstack *)malloc(sizeof(struct intstack));
		new->next=loop_input_stack;
		new->value=newreg();
		loop_input_stack=new;
		move(consumer_reg(),new->value);

		element=push_regs;
		if(element!=(struct intstack *)NULL) {
			while(element->next!=(struct intstack *)NULL) {
				element=element->next;
			}
		}
		
		new_push_reg=(struct intstack *)malloc(sizeof(struct intstack));
		new_push_reg->value=new->value;
		new_push_reg->next=(struct intstack *)NULL;
		if(element==(struct intstack *)NULL) {
			push_regs=new_push_reg;
		}
		else {
			element->next=new_push_reg; 
		}
	}

	while(push_regs!=(struct intstack *)NULL) {
		regstack_push(push_regs->value);
		element=push_regs;
		push_regs=push_regs->next;
		(void)free(element);
	}
}

/* loop_registers: move values that have been produced at the end of the loop and
 * should flow back to the beginning of the loop into the registers previously
 * reserved by set_loop_registers()
 */
void loop_registers(treenode *node) {
	int a,b;
	struct intstack *element;
	int input_reg;
	int reg;

	struct intstack *reg_element;
	struct intstack *previous;
	struct intstack *loop_previous;
	
	int values=int_pop(&loop_input_values);

	for(a=0;a<values;a++) {
		element=loop_input_stack;
		if(element==(struct intstack *)NULL) {
			fprintf(stderr,"Internal error 12\n");
			exit(5);
		}
		for(b=0;b<values-a-1;b++) {
			element=element->next;
			if(element==(struct intstack *)NULL) {
				fprintf(stderr,"Internal error 13\n");
				exit(5);
			}
		}

		reg_element=reg_stack;
		if(reg_element==(struct intstack *)NULL) {
			fprintf(stderr,"Internal error 14\n");
			exit(5);
		}
		previous=(struct intstack *)NULL;
		for(b=1;b<values-a;b++) {
			previous=reg_element;
			reg_element=reg_element->next;
			if(reg_element==(struct intstack *)NULL) {
				fprintf(stderr,"Internal error 15\n");
				exit(5);
			}
		}

		reg=reg_element->value;
                if(reg<0) {
			reg=-reg;
			fprintf(output,"\taddq $sp, 8, $sp\n");
		}
		if(previous==(struct intstack *)NULL) {
			reg_stack=reg_element->next;
		}
		else {
			previous->next=reg_element->next;
		}

		input_reg=freereg_noid(reg);
		move(input_reg,element->value);

	}
	for(a=0;a<values;a++) {
		int_pop(&loop_input_stack);
	}
}

/* push_loop_registers: put the registers that hold the output values of a loop
 * on top of the registers that hold the values to be consumed by the next
 * statement
 */
void push_loop_registers(void) {
	int reg;
	int a,b;

	struct intstack *element;
	struct intstack *previous;

	for(a=valuecount;a>0;a--) {
		element=loop_stack;
		previous=(struct intstack *)NULL;

		if(element==(struct intstack *)NULL) {
			fprintf(stderr,"Internal error 16\n");
			exit(5);
		}
		for(b=1;b<a;b++) {
			previous=element;
			element=element->next;
			if(element==(struct intstack *)NULL) {
				fprintf(stderr,"Internal error 17\n");
				exit(5);
			}
		}

		reg=element->value;
		if(previous==(struct intstack *)NULL) {
			loop_stack=element->next;
		}
		else {
			previous->next=element->next;
		}
		(void)free(element);

		regstack_push(reg);
	}
}

/* loop_output: put the values that are flowing from left to right at an
 * assignment operator at the end of an ifexit-branch into the registers
 * previously reserved for the output of the if block by set_loop_registers()
 */
void loop_output(void) {
	int reg;
	int a,b;

	struct intstack *element;
	struct intstack *previous;

	int valuecount_save=valuecount;
	for(a=valuecount_save;a>0;a--) {
		element=loop_stack;

		if(element==(struct intstack *)NULL) {
			fprintf(stderr,"Internal error 18\n");
			exit(5);
		}
		for(b=1;b<a;b++) {
			element=element->next;
			if(element==(struct intstack *)NULL) {
				fprintf(stderr,"Internal error 19\n");
				exit(5);
			}
		}

		move(consumer_reg(),element->value);
	}
}

/* save_registers: move values of the registers from $9 to $16 onto the stack
 */
void save_registers() {
	int a;

	for(a=9;a<16;a++) {
		fprintf(output,"\tlda $sp, -8($sp)\n");
		fprintf(output,"\tstq $%i, 0($sp)\n",a);
	}
}

/* restore_registers: restore the values of the registers from $9 to $16 from
 * the stack (these values have previously been saved by save_registers()
 */
void restore_registers() {
	int a;

	for(a=15;a>8;a--) {
		fprintf(output,"\tldq $%i, 0($sp)\n",a);
		fprintf(output,"\taddq $sp, 8, $sp\n");
	}
}

/* andjump: when a part of a conjunction evaluates to false, processing this
 * conjunction can be interrupted because this conjunction will never evaluate
 * to true
 * boolean expressions can only appear in an expression of an if statement so
 * when the conjunction processed currently is on top level of this boolean
 * expression we can immediately jump to the else-branch of the if-statement
 * or the end of the ifexit statement
 * when there is an negation around the conjunction, we may only jump to the end
 * of processing the conjunction
 */
void andjump(void) {
	if(reg_stack==(struct intstack *)NULL) {
		fprintf(stderr,"Internal error 20\n");
		exit(5);
	}
	if(negcounter==0) {
		fprintf(output,"\tbeq $%i, ELSE_%i\n",reg_stack->value,ifcounter);
	}
	else {
		fprintf(output,"\tbeq $%i,ENDNOT_%i\n",POP,negcounter);
	}
}

/* init: set initial values for global variables */
void init(void) {
	/* see code_gen.h about information about these variables */
	int a;

	for(a=0;a<32;a++) {
		freereg(a);
	}

	func_par_index=0;
	reg_data_stack=(struct reg_data *)NULL;
	ifcounter=0;
	loopcounter=0;

	maxif=0;
	maxloop=0;
	maxneg=0;

	if_stack=(struct intstack *)NULL;
	loop_stack=(struct intstack *)NULL;
	loop_input_stack=(struct intstack *)NULL;

	reg_stack=(struct intstack *)NULL;

	loop_input_values=(struct intstack *)NULL;
	ifstack=(struct intstack *)NULL;
	loopstack=(struct intstack *)NULL;
	negstack=(struct intstack *)NULL;

	negcounter=0;

	register_assignments=(struct register_assignment *)NULL;

	standard_output=1;
	output=stdout;
	null=fopen("/dev/null","a");
}

/* new_identifier_reg: add new register assignment for given identifier and register */
int new_identifier_reg(char *name, int reg) {
	struct register_assignment *new;
	struct register_assignment *first;
	struct register_assignment *previous;

	first=register_assignments;
	while(first!=(struct register_assignment *)NULL) {
		if(strcmp(first->name,name)==0) {
			fprintf(stderr,"Duplicate identifier\n");
			exit(5);
		}
		first=first->next;
	}
	new=(struct register_assignment *)malloc(sizeof(struct register_assignment));
	new->reg=reg;
	new->name=name;
	new->next=(struct register_assignment *)NULL;

	first=register_assignments;
	previous=(struct register_assignment *)NULL;
	while(first!=(struct register_assignment *)NULL) {
		previous=first;
		first=first->next;
	}
	if(previous==(struct register_assignment *)NULL) {
		register_assignments=new;
	}
	else {
		previous->next=new;
	}

	return reg;
}

/* new_identifier: get new register assignment for given identifier
 * register will be reserved
 * index of chosen register will be returned
 */
int new_identifier(char *name) {
	int reg;
	struct register_assignment *new;
	struct register_assignment *first;
	struct register_assignment *previous;

	first=register_assignments;
	while(first!=(struct register_assignment *)NULL) {
		if(strcmp(first->name,name)==0) {
			fprintf(stderr,"Duplicate identifier\n");
			exit(5);
		}
		first=first->next;
	}
	/* identifier are stored in safe registers */
	reg=newreg_safe();
	
	new=(struct register_assignment *)malloc(sizeof(struct register_assignment));
	new->reg=reg;
	new->name=name;
	new->next=(struct register_assignment *)NULL;

	first=register_assignments;
	previous=(struct register_assignment *)NULL;
	while(first!=(struct register_assignment *)NULL) {
		previous=first;
		first=first->next;
	}
	if(previous==(struct register_assignment *)NULL) {
		register_assignments=new;
	}
	else {
		previous->next=new;
	}

	return reg;
}

/* get_register: get the register that stores the value of the given identifier */
int get_register(char *name) {
	struct register_assignment *first;
	struct register_assignment *previous;
	int reg=-1;

	first=register_assignments;
	previous=(struct register_assignment *)NULL;
	while(first!=(struct register_assignment *)NULL) {
		if(strcmp(first->name,name)==0) {
			reg=first->reg;
			break;
		}
		previous=first;
		first=first->next;
	}

	return reg;
}

/* free_all_registers: reset all register assignments and free all registers
 * necessary at the end of a function so that the next function can safely take use
 * of all the registers ;)
 */
void free_all_registers(void) {
	int a;
	
	while(register_assignments!=(struct register_assignment *)NULL) {
		drop_identifier(register_assignments->name);
	}

	for(a=0;a<32;a++) {
		freereg(a);
	}
}

/* drop_identifier: remove register assignment for a given identifier and
 * free the register belonging to this identifier
 */
int drop_identifier(char *name) {
	struct register_assignment *first;
	struct register_assignment *previous;
	int reg=-1;

	first=register_assignments;
	previous=(struct register_assignment *)NULL;
	while(first!=(struct register_assignment *)NULL) {
		if(strcmp(first->name,name)==0) {
			reg=first->reg;
			if(previous==(struct register_assignment *)NULL) {
				register_assignments=first->next;
			}
			else {
				previous->next=first->next;
			}
			(void)free(first);
			freereg(reg);
			break;
		}
		previous=first;
		first=first->next;
	}

	return reg;
}

/* freereg: free register so that it can be used again later in the program */
int freereg(int reg) {
	reg_used[reg]=0;

	return reg;
}

/* freereg_noid: free register only if it does not store an identifier's value and
 * if it is not used for transferring values from the end of a loop back to its
 * beginning (i.e. it is listed in the stack loop_input_values) */
int freereg_noid(int reg) {
	struct register_assignment *element=register_assignments;
	struct intstack *loop_values=loop_input_stack;

	while(loop_values!=(struct intstack *)NULL) {
		if(loop_values->value==reg) {
			return reg;
		}
		loop_values=loop_values->next;
	}
	
	while(element!=(struct register_assignment *)NULL) {
		if(element->reg==reg) {
			return reg;
		}
		element=element->next;
	}

	return freereg(reg);
}

/* newreg: reserves an unused register for storing a value and returns this
 * register's index
 */
int newreg(void) {
	int a=1;

	while(1) {
		if(!reg_used[a]) {
			reg_used[a]=1;
			return a;
		}
		if(a==8) {
			a=22;
		}
		else if(a==25) {
			a=9;
		}
		else if(a==14) {
			/* first step: search for least recently used register (bottom of stack)
			 * that has not been saved onto stack */
			struct intstack *stackpointer=reg_stack;
			
			/* TODO: NULL ? */
			while(stackpointer->next!=(struct intstack *)NULL && stackpointer->next->value>0) { 
				stackpointer=stackpointer->next;
			}

			/* second step: put value of this register onto stack */
			fprintf(output,"\tlda $sp, -8($sp)\n");
			fprintf(output,"\tstq $%i, 0($sp)\n",stackpointer->value);

			/* third step: set flag to 1 */
			stackpointer->value=-stackpointer->value;

			/* no free register found */
			return -stackpointer->value;
		}
		else {
			a++;
		}
	}
}

/* newreg_safe: reserves a safe register (from $9 to $14) for storing a value
 * (usually an identifier) and returns this register's index
 */
int newreg_safe(void) {
	int a;
	int start, end;
	
	if(has_call) {
		start=9;
		end=14;
	}
	else {
		start=16;
		end=21;
	}
	
	for(a=start;a<=end;a++) {
		if(!reg_used[a]) {
			reg_used[a]=1;
			return a;
		}
	}

	/* no free safe register found */
	fprintf(stderr,"Too many registers in use\n");
	exit(5);
}

/* function_header: prints the header of a function with the given name */
void function_header(char *name, int has_func_call) {
	fprintf(output,"\t.globl %s\n\t.ent %s\n\n%s:\n",name,name,name);
	has_call=has_func_call;
	if(has_call) { 
		save_registers();
	}
	func_par_index=0;
}

/* function_footer: prints the footer of a function with the given name */
void function_footer(char *name) {
	move(regstack_pop(),0);
	if(has_call) {
		restore_registers();
	} 
	fprintf(output,"\tret\t\n\t.end %s\n\n",name,name,name,name);
	free_all_registers();
}

/* compiler_info: prints info about the compiler, printed out one at the very
 * beginning of the program
 */
void compiler_info(void) {
	fprintf(output,"# Compiler written by Paul Staroch\n");
	fprintf(output,"# Uebersetzerbau, SS 2006\n");
	fprintf(output,"# Send bug reports to /dev/null\n\n");
}

/* function_parameter: processes one parameter to a function by storing its
 * value in a register and save the assignment between the identifier and this
 * register
 */
void function_parameter(char *name) {
	func_par_index++;
	if(has_call) {
		reg1=new_identifier(name);
		move(func_par_index+15,reg1);
	}
	else {
		reg1=func_par_index+15;
		(void)new_identifier_reg(name, reg1);
		reg_used[reg1]=1;
	}
}

/* move: prints a mov-statement, only when reg1 and reg2 are not equal */
void move(int reg1, int reg2) {
	if(reg1!=reg2) {
		fprintf(output,"\tmov $%i, $%i\n",reg1,reg2);
	}
}

/* enter_deleted_block: starts a block which contents will not be put to the
 * standard output due to this block being discarded due to optimization
 */
void enter_deleted_block(void) {
	standard_output--;

	if(standard_output==0) {
		output=null;
	}
}

/* exit_deletec_block: ends a block which containts will not be put to the
 * standard output due to this block being discarded due to optimization
 */
void exit_deleted_block(void) {
	standard_output++;

	if(standard_output==1) {
		output=stdout;
	}
}

