/* Uebersetzerbau, SS 2006
 * Paul Staroch, 0425426
 * Optimierende Codeerzeugung
 *
 * C file for maintaining symbol tables
 */

#include <string.h>
#include <malloc.h>
#include <stdio.h>
#include "symbol_table.h"

/* list of symbol tables (for destructing them at the end of
 * program execution)
 */
struct tables *symbol_tables=(struct tables *)NULL;

/* number of errors found at statical analysis */
extern int attributed_errors;

/* check_symbol_count: checks if there are only six variables
 * visible at the same time
 */
void check_symbol_count(struct symbol_t *table) {
	int symbol_count=0;

	struct symbol_t *element=table;
	while(element!=(struct symbol_t *)NULL) {
		symbol_count++;
		element=element->next;
	}

	if(symbol_count>6) {
		(void)fprintf(stderr,"Too many identifiers visible\n");
		(void)fflush(stdout);
		attributed_errors++;
	}
}

/* remove_all_lists: remove all symbol tables that have been
 * created during the static analysis and free the memory
 * used by them
 */
void remove_all_lists(void) {
	struct tables *element;
	struct tables *next_element;

#ifdef DEBUG
	(void)printf("Deleting all symbol tables...\n");
	(void)fflush(stdout);
#endif /* DEBUG */

	if((struct tables *)NULL==symbol_tables) {
		return;
	}
	element=symbol_tables;
	next_element=symbol_tables->next;
	while((struct tables *)NULL!=element) {
		next_element=element->next;
		free_table(element->table);
		(void)free(element->table);
		(void)free(element);
		element=next_element;
	}
}

/* add_to_table_list: adds the given symbol table to the list of all symbol
 * tables */
int add_to_table_list(struct symbol_t *new_table) {
	struct tables *new_element;
	struct tables *element;

#ifdef DEBUG
	(void)printf("Adding new entry to list of symbol tables...\n");
	(void)fflush(stdout);
#endif /* DEBUG */
	
	if((new_element=(struct tables *)malloc(sizeof(struct tables)))==NULL) {
		return -1;
	}
	new_element->table=new_table;
	new_element->next=(struct tables *)NULL;

	element=symbol_tables;
	if((struct tables *)NULL==symbol_tables) {
		symbol_tables=new_element;
		return 0;
	}
	while((struct tables *)NULL!=element->next) {
		element=element->next;
	}
	element->next=new_element;
	return 0;
}
	
/* new_table: initialize a new symbol_table */
struct symbol_t *new_table(void) {
#ifdef DEBUG
	(void)printf("Initializing new symbol table...\n");
	(void)fflush(stdout);
#endif /* DEBUG */
	return (struct symbol_t *)NULL;
}

/* clone_table: create a new symbol table and insert all the
 * elements from an existing one into it
 */
struct symbol_t *clone_table(struct symbol_t *table) {
	struct symbol_t *element;
	struct symbol_t *new_tablex;

#ifdef DEBUG
	(void)printf("Cloning symbol table...\n");
	(void)fflush(stdout);
#endif /* DEBUG */

	element=table;
	new_tablex=new_table();
	while((struct symbol_t *)NULL!=element) {
		/* check return value */
		new_tablex=table_add_symbol(new_tablex,element->identifier);
		element=element->next;
	}

#ifdef DEBUG
	(void)printf("Cloning completed...\n");
	(void)fflush(stdout);
#endif /* DEBUG */

	return new_tablex;
}

/* table_add_symbol: add the given identifier to the given symbol table */
struct symbol_t *table_add_symbol(struct symbol_t *table, char *identifier) {
	struct symbol_t *element;
	struct symbol_t *new_element;

#ifdef DEBUG
	(void)printf("Adding identifier %s to symbol table...\n", identifier);
	(void)fflush(stdout);
#endif /* DEBUG */

	if(check_symbol_free(table,identifier)==0) {
		return table;
	}
	
	if((new_element=(struct symbol_t *)malloc(sizeof(struct symbol_t)))==NULL) {
		return (struct symbol_t *)NULL;
	}
	new_element->next=(struct symbol_t *)NULL;
	if((new_element->identifier=(char *)malloc(sizeof(char)*strlen(identifier)+1))==NULL) {
		(void)free(new_element);
		return (struct symbol_t *)NULL;
	}
	(void)strcpy(new_element->identifier,identifier);
	
	if((struct symbol_t *)NULL==table) {
		/* check return value */
		(void)add_to_table_list(new_element);
		return new_element;
	}
	element=table;

	while((struct symbol_t *)NULL!=element->next) {
		element=element->next;
	}

	element->next=new_element;
	
	check_symbol_count(table);
	
	return table;
}

/* table_remove_symbol: remove the given identifier from the given symbol table
 * (only it is listed in that symbol table)
 */
struct symbol_t *table_remove_symbol(struct symbol_t *table, char *identifier) {
	struct symbol_t *element;
	struct symbol_t *previous_element;
	struct symbol_t *new_element;

#ifdef DEBUG
	(void)printf("Removing identifier %s from symbol table...\n",identifier);
	(void)fflush(stdout);
#endif /* DEBUG */
	
	if((struct symbol_t *)NULL==table) {
		return (struct symbol_t *)NULL;
	}

	previous_element=(struct symbol_t *)NULL;
	element=table;
	
	while((struct symbol_t *)NULL!=element->next) {
		if(strcmp(element->identifier,identifier)==0) {
			if((struct symbol_t *)NULL==previous_element) {
				new_element=element->next;
				(void)free(element->identifier);
				(void)free(element);
			}
			previous_element->next=element->next;
			(void)free(element->identifier);
			(void)free(element);
			return table;
		}
		previous_element=element;
		element=element->next;
	}

	return table;
}

/* check_symbol_free: returns a Boolean value that tells you if an identifier is not
 * listed in an symbol table
 */
int check_symbol_free(struct symbol_t *table, char *identifier) {
	if(table_lookup(table, identifier)!=(struct symbol_t *)NULL) {
		(void)fprintf(stderr,"Identifier %s redefined in its scope\n",identifier);
		(void)fflush(stderr);
		attributed_errors++;
		return 0;
	}
	return 1;
}

/* check_symbol_defined: returns a Boolean value that tells you if an identifier is
 * listed in an symbol table
 */
int check_symbol_defined(struct symbol_t *table, char *identifier) {
	if(table_lookup(table, identifier)==(struct symbol_t *)NULL) {
		(void)fprintf(stderr,"Identifier %s undefined or invisible\n",identifier);
		(void)fflush(stderr);
		attributed_errors++;
		return 0;
	}
	return 1;
}

/* table_lookup: performs a search for an identifier in a symbol_table */
struct symbol_t *table_lookup(struct symbol_t *table, char *identifier) {
	struct symbol_t *element;

#ifdef DEBUG
	(void)printf("Looking for identifier %s in symbol table...\n",identifier);
	(void)fflush(stdout);
#endif /* DEBUG */

	element=table;

	if((struct symbol_t *)NULL==table) {
		return (struct symbol_t *)NULL;
	}
	
	if(strcmp(element->identifier,identifier)==0) {
		return element;
	}
	
	while((struct symbol_t *)NULL!=element->next) {
		element=element->next;
		if(strcmp(element->identifier,identifier)==0) {
			return element;
		}
	}

	return (struct symbol_t *)NULL;
}

/* table_merge: creates a new symbol table that holds all the elements that occur in
 * the two symbol tables */
struct symbol_t *table_merge(struct symbol_t *table, struct symbol_t *to_add) {
	struct symbol_t *element;
	struct symbol_t *new_table=clone_table(table);
	
#ifdef DEBUG
	(void)printf("Merging symbol tables...\n");
	(void)fflush(stdout);
#endif

	element=to_add;
	while(element!=(struct symbol_t *)NULL) {
		table_add_symbol(new_table,element->identifier);
		element=element->next;
	}

	check_symbol_count(new_table);
	
	return new_table;
}

/* free_table: destroy the given symbol table and free the memory its elements use */
void free_table(struct symbol_t *table) {
	struct symbol_t *element;
	struct symbol_t *next_element;

#ifdef DEBUG
	(void)printf("Removing symbol table...\n");
	(void)fflush(stdout);
#endif /* DEBUG */

	if((struct symbol_t *)NULL==table) {
		return;
	}
	element=table;
	next_element=table->next;
	while((struct symbol_t *)NULL!=element) {
		next_element=element->next;
		(void)free(element->identifier);
		(void)free(element);
		element=next_element;
	}
}

#ifdef DEBUG
/* print_table: print out all the identifier the given symbol table contains */
void print_table(struct symbol_t *table) {
	struct symbol_t *element=table;

	while(element!=(struct symbol_t *)NULL) {
		(void)printf("%s\n",element->identifier);
		(void)fflush(stdout);
		element=element->next;
	}
}
#endif /* DEBUG */
		
