package at.ac.tuwien.dslab.rmi.server;

import java.net.URI;
import java.rmi.Naming;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import at.ac.tuwien.dslab.rmi.common.BuddyAlreadyExistsException;
import at.ac.tuwien.dslab.rmi.common.BuddyAuthenticationException;
import at.ac.tuwien.dslab.rmi.common.BuddyNotFoundException;
import at.ac.tuwien.dslab.rmi.common.BuddyNotOnlineException;
import at.ac.tuwien.dslab.rmi.common.SkyNetException;
import at.ac.tuwien.dslab.rmi.common.interfaces.IBuddy;
import at.ac.tuwien.dslab.rmi.common.interfaces.ISkyNetListener;
import at.ac.tuwien.dslab.rmi.common.interfaces.ISkyNetManager;
import at.ac.tuwien.dslab.rmi.db.SkyNetDBException;
import at.ac.tuwien.dslab.rmi.db.interfaces.ISkyNetDBService;

public class SkyNetManagerImpl extends UnicastRemoteObject implements ISkyNetManager, ISkyNetListener {
	
	private static final long serialVersionUID = -592325121348910288L;
	
	private Vector<ISkyNetListener> buddyListener = new Vector<ISkyNetListener>();
	private HashMap<IBuddy, ISkyNetListener> allBuddies = new HashMap<IBuddy, ISkyNetListener>();
	private HashMap<IBuddy, ISkyNetListener> localBuddies = new HashMap<IBuddy, ISkyNetListener>();
	private ISkyNetDBService dbService;
	private Vector<String> sourceIDs = new Vector<String>(); // log received sourceIDs
	private String sourceRmiURI;
	
	public SkyNetManagerImpl(URI sourceRmiURI, URI dbServiceURI, URI[] buddyManagerURIs) throws RemoteException {
		super();
		// initialize data structures for server
		this.sourceRmiURI = sourceRmiURI.toString();
		try {
			// lookup DB service
			dbService = (ISkyNetDBService) Naming.lookup(dbServiceURI.toString());
		} catch (Exception e) {
			System.err.println(e);
		}
		// connections to other RMI servers (via own thread)
		new ConnectThread(this, buddyManagerURIs).start();
	}
	
	public void addSkyNetListener(ISkyNetListener listener) throws RemoteException {
		System.out.println(sourceRmiURI + ": addSkyNetListener");
		buddyListener.add(listener);
	}

	public IBuddy findBuddy(String nickname) throws RemoteException {
		System.out.println(sourceRmiURI + ": findBuddy");
		IBuddy buddy = null;
		try {
			buddy = dbService.findBuddyByNickname(nickname);
		} catch (SkyNetDBException e) {
			System.err.println(e);
		}
		return buddy;
	}

	public List<IBuddy> getOnlineBuddies(IBuddy requestor) throws RemoteException, BuddyAuthenticationException {
		System.out.println(sourceRmiURI + ": getOnlineBuddies");
		List<IBuddy> onlineBuddies = new Vector<IBuddy>();
		IBuddy buddy = findBuddy(requestor.getNickname());
		if (!requestor.getNickname().equals(buddy.getNickname()) ||
			!requestor.getPassword().equals(buddy.getPassword()) ||
			!isBuddyOnline(buddy.getNickname())) {
			throw new BuddyAuthenticationException("buddy not authenticated");
		}
		Iterator it = allBuddies.keySet().iterator();
		while (it.hasNext()) {
			buddy = (IBuddy) it.next();
			onlineBuddies.add(buddy);
		}
		return onlineBuddies;
	}

	public boolean isBuddyOnline(String nickname) throws RemoteException {
		System.out.println(sourceRmiURI + ": isBuddyOnline");
		IBuddy buddy = findBuddy(nickname);
		if (allBuddies.get(buddy) != null) {
			return true;
		}
		return false;
	}

	public IBuddy login(String nickname, String password, ISkyNetListener listener) throws BuddyAuthenticationException, BuddyNotFoundException, RemoteException {
		System.out.println(sourceRmiURI + ": login");
		IBuddy buddy = findBuddy(nickname);
		if (buddy == null) {
			throw new BuddyNotFoundException("buddy not found");
		}
		if (nickname.equals(buddy.getNickname()) && password.equals(buddy.getPassword())) {
			allBuddies.put(buddy, listener);
			localBuddies.put(buddy, listener);
			// send notifications to servers ...
			String sID = generateSourceID();
			sourceIDs.add(sID);
			Iterator it = buddyListener.iterator();
			while (it.hasNext()) {
				ISkyNetListener server = (ISkyNetListener) it.next();
				server.buddyLoggedIn(sID, buddy, listener);
			}
			return buddy;
		}
		throw new BuddyAuthenticationException("wrong nickname or password");
	} // end login

	public void logout(String nickname, String password) throws BuddyAuthenticationException, BuddyNotFoundException, RemoteException {
		System.out.println(sourceRmiURI + ": logout");
		IBuddy buddy = findBuddy(nickname);
		if (localBuddies.get(buddy) == null) {
			throw new BuddyNotFoundException("buddy not online");
		}
		if (nickname.equals(buddy.getNickname()) && password.equals(buddy.getPassword())) {
			allBuddies.remove(buddy);
			localBuddies.remove(buddy);
			// send notifications to servers ...
			String sID = generateSourceID();
			sourceIDs.add(sID);
			Iterator it = buddyListener.iterator();
			while (it.hasNext()) {
				ISkyNetListener server = (ISkyNetListener) it.next();
				server.buddyLoggedOut(sID, buddy);
			}
			return;
		}
		throw new BuddyAuthenticationException("wrong nickname or password");
	} // end logout

	public IBuddy register(IBuddy buddy) throws RemoteException, BuddyAlreadyExistsException {
		System.out.println(sourceRmiURI + ": register");
		try {
			IBuddy buddy2 = dbService.findBuddyByNickname(buddy.getNickname());
			if (buddy2 != null) {
				throw new BuddyAlreadyExistsException("buddy already registered");
			}
			dbService.insertBuddy(buddy);
		} catch (SkyNetDBException e) {
			System.err.println(e);
		}
		return buddy;
	} // end register

	public void removeSkyNetListener(ISkyNetListener listener) throws RemoteException {
		System.out.println(sourceRmiURI + ": removeSkyNetListener");
		buddyListener.remove(listener);
	}

	public List<IBuddy> searchBuddy(String searchString) throws RemoteException, SkyNetException {
		System.out.println(sourceRmiURI + ": searchBuddy");
		List<IBuddy> buddies = null;
		try {
			buddies = dbService.searchBuddy(searchString);
		} catch (SkyNetDBException e) {
			System.err.println(e);
		}
		return buddies;
	}

	private void testTransferBuddies(IBuddy sender, String receiver) throws RemoteException, BuddyNotOnlineException, BuddyAuthenticationException {
		IBuddy buddy = findBuddy(sender.getNickname());
		if (!buddy.getNickname().equals(sender.getNickname()) ||
			!buddy.getPassword().equals(sender.getPassword()) ||
			!isBuddyOnline(buddy.getNickname())) {
			throw new BuddyAuthenticationException("sender not authenticated");
		}
		if (!isBuddyOnline(receiver)) {
			throw new BuddyNotOnlineException(receiver);
		}
	} // end testTransferBuddies
	
	public void sendFile(IBuddy sender, String receiver, String filename, byte[] data) throws RemoteException, BuddyNotOnlineException, BuddyAuthenticationException {
		System.out.println(sourceRmiURI + ": sendFile");
		IBuddy buddy = findBuddy(receiver);
		testTransferBuddies(sender, receiver);
		allBuddies.get(buddy).fileReceived(sender.getNickname(), receiver, filename, data);
	} // end sendFile

	public void sendMessage(IBuddy sender, String receiver, String message) throws RemoteException, BuddyAuthenticationException, BuddyNotOnlineException {
		System.out.println(sourceRmiURI + ": sendMessage");
		IBuddy buddy = findBuddy(receiver);
		testTransferBuddies(sender, receiver);
		allBuddies.get(buddy).messageReceived(sender.getNickname(), receiver, message);
	}

	public IBuddy unregister(String nickname, String password) throws RemoteException, BuddyNotFoundException, BuddyAuthenticationException {
		System.out.println(sourceRmiURI + ": unregister");
		IBuddy buddy = null;
		try {
			buddy = dbService.findBuddyByNickname(nickname);
			if (buddy == null) {
				throw new BuddyNotFoundException("buddy not found");
			}
			if (nickname.equals(buddy.getNickname()) && password.equals(buddy.getPassword())) {
				dbService.removeBuddy(buddy);
				return buddy;
			}
		} catch (SkyNetDBException e) {
			System.err.println(e);
		}
		throw new BuddyAuthenticationException("wrong nickname or password");
	} // end unregister

	public IBuddy updateBuddy(IBuddy buddy) throws RemoteException, BuddyAuthenticationException, BuddyNotFoundException {
		System.out.println(sourceRmiURI + ": updateBuddy");
		IBuddy buddy2 = null;
		try {
			buddy2 = dbService.findBuddyByNickname(buddy.getNickname());
			if (buddy2 == null) {
				throw new BuddyNotFoundException("buddy not found");
			}
			if (buddy2.getNickname().equals(buddy.getNickname()) &&
				buddy2.getPassword().equals(buddy.getPassword()) &&
				isBuddyOnline(buddy.getNickname())) {
				dbService.updateBuddy(buddy);
				return buddy;
			}
		} catch (SkyNetDBException e) {
			System.err.println(e);
		}
		throw new BuddyAuthenticationException("wrong nickname or password");
	} // updateBuddy

	private String generateSourceID() {
		Calendar cal = Calendar.getInstance();
		return sourceRmiURI + ":" + cal.getTimeInMillis();
	}
	
	// from ISkyNetListener (next 4 methods)
	public void buddyLoggedIn(String sourceID, IBuddy buddy, ISkyNetListener listener) throws RemoteException {
		// not implemented: check of source IDs
		System.out.println(sourceRmiURI + ": buddyLoggedIn");
		// store buddy as being logged in
		allBuddies.put(buddy, listener);
		// send notifications to servers ...
		Iterator it = buddyListener.iterator();
		if (!sourceIDs.contains(sourceID)) {
			sourceIDs.add(sourceID);
			while (it.hasNext()) {
				ISkyNetListener server = (ISkyNetListener) it.next();
				server.buddyLoggedIn(sourceID, buddy, listener);
			}
		}
		// ... and directly logged-in buddies (with the logged-in buddy on the contact list)
		it = localBuddies.keySet().iterator();
		while (it.hasNext()) {
			IBuddy buddy2 = (IBuddy) it.next();
			if (buddy2.getContactList().findBuddyByNick(buddy.getNickname()) != null) {
				localBuddies.get(buddy2).buddyLoggedIn(sourceID, buddy, listener);
			}
		}
	} // end buddyLoggedIn

	public void buddyLoggedOut(String sourceID, IBuddy buddy) throws RemoteException {
		// not implemented: check of source IDs
		System.out.println(sourceRmiURI + ": buddyLoggedOut");
		// remove buddy from list of online buddies
		allBuddies.remove(buddy);
		// send notification to servers ...
		Iterator it = buddyListener.iterator();
		if (!sourceIDs.contains(sourceID)) {
			sourceIDs.add(sourceID);
			while (it.hasNext()) {
				ISkyNetListener server = (ISkyNetListener) it.next();
				server.buddyLoggedOut(sourceID, buddy);
			}
		}
		// ... and directly logged-in buddies (with the logged-in buddy on the contact list)
		it = localBuddies.keySet().iterator();
		while (it.hasNext()) {
			IBuddy buddy2 = (IBuddy) it.next();
			if (buddy2.getContactList().findBuddyByNick(buddy.getNickname()) != null) {
				localBuddies.get(buddy2).buddyLoggedOut(sourceID, buddy);
			}
		}
	} // end buddyLoggedOut

	public void fileReceived(String senderNickname, String receiverNickname, String filename, byte[] data) throws RemoteException {
		// ignore
	}

	public void messageReceived(String senderNickname, String receiverNickname, String msg) throws RemoteException {
		// ignore
	}

}
